(function(){class BaseMonitor {
  lastActivityTime;
  constructor() {
    this.lastActivityTime = /* @__PURE__ */ new Date();
  }
  updateActivity() {
    this.lastActivityTime = /* @__PURE__ */ new Date();
  }
}
class MuteStateMonitor extends BaseMonitor {
  observer;
  isObserving;
  lastKnownState;
  currentObservedElement;
  debounceTimer;
  stateCheckInterval;
  isProcessing;
  muteStateHistory;
  lastStateChangeTime;
  constructor() {
    super();
    this.observer = null;
    this.isObserving = false;
    this.lastKnownState = null;
    this.currentObservedElement = null;
    this.debounceTimer = null;
    this.stateCheckInterval = null;
    this.isProcessing = false;
    this.muteStateHistory = [];
    this.lastStateChangeTime = 0;
  }
  start() {
    this.startContinuousMonitoring();
    this.startPeriodicValidation();
    this.startDOMChangeMonitoring();
  }
  stop() {
    if (this.observer) {
      this.observer.disconnect();
      this.observer = null;
    }
    if (this.debounceTimer) {
      clearTimeout(this.debounceTimer);
      this.debounceTimer = null;
    }
    if (this.stateCheckInterval) {
      clearInterval(this.stateCheckInterval);
      this.stateCheckInterval = null;
    }
    this.isObserving = false;
    this.currentObservedElement = null;
    this.isProcessing = false;
  }
  // Start continuous monitoring with multiple detection methods
  startContinuousMonitoring() {
    setInterval(() => {
      if (!this.isProcessing) {
        this.checkMuteStateFast();
      }
    }, 200);
    setInterval(() => {
      if (!this.isProcessing) {
        this.checkMuteStateComprehensive();
      }
    }, 1e3);
    setInterval(() => {
      if (!this.isProcessing) {
        this.validateMuteState();
      }
    }, 3e3);
  }
  // Start periodic validation to catch missed changes
  startPeriodicValidation() {
    this.stateCheckInterval = setInterval(() => {
      this.performStateValidation();
    }, 5e3);
  }
  // Start DOM change monitoring
  startDOMChangeMonitoring() {
    const documentObserver = new MutationObserver((mutations) => {
      let shouldCheck = false;
      mutations.forEach((mutation) => {
        if (mutation.type === "childList") {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType === Node.ELEMENT_NODE) {
              const element = node;
              if (this.isMuteButton(element) || element.querySelector('[data-is-muted], [aria-label*="microphone"], [aria-label*="mute"]')) {
                shouldCheck = true;
              }
            }
          });
        }
      });
      if (shouldCheck) {
        setTimeout(() => {
          this.checkMuteStateComprehensive();
        }, 100);
      }
    });
    documentObserver.observe(document.body, {
      childList: true,
      subtree: true
    });
  }
  // Fast mute state check for immediate response
  checkMuteStateFast() {
    try {
      const muteButton = this.findMuteButton();
      if (muteButton) {
        const isMuted = this.determineMuteState(muteButton);
        if (isMuted !== null && isMuted !== this.lastKnownState) {
          this.handleMuteStateChange(isMuted, "fast-check");
        }
      }
    } catch (error) {
    }
  }
  // Comprehensive mute state check
  checkMuteStateComprehensive() {
    if (this.isProcessing) return;
    this.isProcessing = true;
    try {
      const muteButtons = this.findAllMuteButtons();
      let bestMuteButton = null;
      let bestMuteState = null;
      let confidence = 0;
      muteButtons.forEach((button) => {
        const muteState = this.determineMuteState(button);
        const buttonConfidence = this.calculateButtonConfidence(button, muteState);
        if (buttonConfidence > confidence) {
          bestMuteButton = button;
          bestMuteState = muteState;
          confidence = buttonConfidence;
        }
      });
      if (bestMuteButton && bestMuteState !== null) {
        this.updateObserver(bestMuteButton);
        if (bestMuteState !== this.lastKnownState) {
          this.handleMuteStateChange(bestMuteState, "comprehensive-check");
        }
      } else {
        this.handleNoMuteButtonFound();
      }
    } catch (error) {
    } finally {
      this.isProcessing = false;
    }
  }
  // Find all possible mute buttons
  findAllMuteButtons() {
    const selectors = [
      "[data-is-muted]",
      "button[data-is-muted]",
      'button[aria-label*="microphone"]',
      'button[aria-label*="mute"]',
      'button[aria-label*="unmute"]',
      '[aria-label*="microphone"]',
      '[aria-label*="mute"]',
      '[aria-label*="unmute"]',
      '[data-tooltip*="microphone"]',
      'button[data-tooltip*="microphone"]',
      'div[role="button"][aria-label*="microphone"]',
      'div[role="button"][data-is-muted]',
      '[jsname="BOHaEe"] button[aria-label*="microphone"]',
      '[jsname="BOHaEe"] button[data-is-muted]'
    ];
    const buttons = [];
    selectors.forEach((selector) => {
      const elements = document.querySelectorAll(selector);
      elements.forEach((element) => {
        if (!buttons.includes(element)) {
          buttons.push(element);
        }
      });
    });
    return buttons;
  }
  // Find the best mute button
  findMuteButton() {
    const buttons = this.findAllMuteButtons();
    for (const button of buttons) {
      const muteState = this.determineMuteState(button);
      if (muteState !== null) {
        return button;
      }
    }
    return null;
  }
  // Calculate confidence score for a button
  calculateButtonConfidence(button, _muteState) {
    let confidence = 0;
    if (button.hasAttribute("data-is-muted")) {
      confidence += 50;
    }
    const ariaLabel = button.getAttribute("aria-label") || "";
    if (ariaLabel.includes("microphone") || ariaLabel.includes("mute")) {
      confidence += 30;
    }
    if (button.tagName === "BUTTON" || button.getAttribute("role") === "button") {
      confidence += 20;
    }
    if (button.closest('[jsname="BOHaEe"]')) {
      confidence += 10;
    }
    return confidence;
  }
  // Check if an element is a mute button
  isMuteButton(element) {
    return element.hasAttribute("data-is-muted") || (element.getAttribute("aria-label")?.includes("microphone") ?? false) || (element.getAttribute("aria-label")?.includes("mute") ?? false) || (element.getAttribute("data-tooltip")?.includes("microphone") ?? false);
  }
  // Handle mute state change with debouncing
  handleMuteStateChange(isMuted, source) {
    const now = Date.now();
    if (now - this.lastStateChangeTime < 100) {
      return;
    }
    this.lastStateChangeTime = now;
    this.muteStateHistory.push(isMuted);
    if (this.muteStateHistory.length > 5) {
      this.muteStateHistory.shift();
    }
    this.lastKnownState = isMuted;
    this.updateActivity();
    this.debounceNotifications(() => {
      this.sendMuteStateNotifications(isMuted, source);
    });
  }
  // Debounce notifications to prevent spam
  debounceNotifications(callback) {
    if (this.debounceTimer) {
      clearTimeout(this.debounceTimer);
    }
    this.debounceTimer = setTimeout(() => {
      callback();
      this.debounceTimer = null;
    }, 100);
  }
  // Send mute state notifications
  sendMuteStateNotifications(isMuted, source) {
    chrome.runtime.sendMessage({
      type: "muteStateChanged",
      isMuted,
      timestamp: (/* @__PURE__ */ new Date()).toISOString(),
      source
    }).catch((_error) => {
    });
    try {
      window.dispatchEvent(new CustomEvent("muteStateChanged", {
        detail: {
          isMuted,
          timestamp: (/* @__PURE__ */ new Date()).toISOString(),
          source
        }
      }));
    } catch (error) {
    }
    try {
      if (window.parent && window.parent !== window) {
        window.parent.postMessage({
          type: "muteStateChanged",
          isMuted,
          timestamp: (/* @__PURE__ */ new Date()).toISOString(),
          source
        }, "*");
      }
    } catch (error) {
    }
  }
  // Handle case when no mute button is found
  handleNoMuteButtonFound() {
    this.isObserving = false;
    if (this.observer) {
      this.observer.disconnect();
      this.observer = null;
    }
    setTimeout(() => {
      this.checkMuteStateComprehensive();
    }, 1e3);
  }
  // Update observer for the current mute button
  updateObserver(muteButton) {
    if (this.currentObservedElement === muteButton) {
      return;
    }
    if (this.observer) {
      this.observer.disconnect();
    }
    this.observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        if (mutation.type === "attributes") {
          const target = mutation.target;
          const isMuted = this.determineMuteState(target);
          if (isMuted !== null && isMuted !== this.lastKnownState) {
            this.handleMuteStateChange(isMuted, "mutation-observer");
          }
        }
      });
    });
    this.observer.observe(muteButton, {
      attributes: true,
      attributeFilter: ["data-is-muted", "aria-label", "aria-pressed", "data-tooltip"]
    });
    this.currentObservedElement = muteButton;
    this.isObserving = true;
  }
  // Validate current mute state
  validateMuteState() {
    try {
      const muteButton = this.findMuteButton();
      if (muteButton) {
        const currentState = this.determineMuteState(muteButton);
        if (currentState !== null && currentState !== this.lastKnownState) {
          this.handleMuteStateChange(currentState, "validation");
        }
      }
    } catch (error) {
    }
  }
  // Perform comprehensive state validation
  performStateValidation() {
    const muteButton = this.findMuteButton();
    if (muteButton) {
      const currentState = this.determineMuteState(muteButton);
      if (currentState !== null) {
        const recentStates = this.muteStateHistory.slice(-3);
        const isConsistent = recentStates.every((state) => state === currentState);
        if (!isConsistent) {
          this.handleMuteStateChange(currentState, "state-correction");
        }
      }
    }
  }
  checkAndNotifyMuteState() {
    const allMicrophoneButtons = document.querySelectorAll(`
      button[aria-label*="microphone"],
      button[aria-label*="mute"],
      button[aria-label*="unmute"],
      [data-is-muted],
      button[data-is-muted],
      [aria-label*="microphone"],
      [aria-label*="mute"],
      [aria-label*="unmute"],
      [data-tooltip*="microphone"],
      [data-tooltip*="mute"],
      [data-tooltip*="unmute"],
      div[role="button"][aria-label*="microphone"],
      div[role="button"][data-is-muted]
    `);
    let muteButton = null;
    let isMuted = false;
    for (let i = 0; i < allMicrophoneButtons.length; i++) {
      const button = allMicrophoneButtons[i];
      const buttonMuteState = this.determineMuteState(button);
      if (buttonMuteState !== null) {
        muteButton = button;
        isMuted = buttonMuteState;
        break;
      }
    }
    if (!muteButton) {
      const unmuteButtons = document.querySelectorAll('button[aria-label*="unmute"], [aria-label*="unmute"]');
      if (unmuteButtons.length > 0) {
        muteButton = unmuteButtons[0];
        isMuted = true;
      }
      if (!muteButton) {
        const muteButtons = document.querySelectorAll('button[aria-label*="mute"], [aria-label*="mute"]');
        if (muteButtons.length > 0) {
          muteButton = muteButtons[0];
          isMuted = false;
        }
      }
    }
    if (muteButton) {
      const shouldSendUpdate = this.lastKnownState !== isMuted || this.lastKnownState === null;
      if (shouldSendUpdate) {
        this.lastKnownState = isMuted;
        this.updateActivity();
        chrome.runtime.sendMessage({
          type: "muteStateChanged",
          isMuted,
          timestamp: (/* @__PURE__ */ new Date()).toISOString(),
          source: "content-script"
        }).catch((_error) => {
        });
        try {
          window.dispatchEvent(new CustomEvent("muteStateChanged", {
            detail: { isMuted, timestamp: (/* @__PURE__ */ new Date()).toISOString() }
          }));
        } catch (error) {
        }
        try {
          if (window.parent && window.parent !== window) {
            window.parent.postMessage({
              type: "muteStateChanged",
              isMuted,
              timestamp: (/* @__PURE__ */ new Date()).toISOString(),
              source: "content-script"
            }, "*");
          }
        } catch (error) {
        }
      }
      if (!this.isObserving || this.observer === null) {
        this.setupObserver(muteButton);
        this.isObserving = true;
      } else {
        if (this.currentObservedElement !== muteButton) {
          this.setupObserver(muteButton);
        }
      }
    } else {
      this.isObserving = false;
      if (this.observer) {
        this.observer.disconnect();
        this.observer = null;
      }
      setTimeout(() => this.checkAndNotifyMuteState(), 1e3);
    }
  }
  // Enhanced mute state determination
  determineMuteState(element) {
    if (element.hasAttribute("data-is-muted")) {
      const dataMuted = element.getAttribute("data-is-muted");
      return dataMuted === "true";
    }
    const ariaLabel = element.getAttribute("aria-label") || "";
    if (ariaLabel.includes("unmute") || ariaLabel.includes("Turn on microphone") || ariaLabel.includes("Enable microphone")) {
      return true;
    }
    if (ariaLabel.includes("mute") || ariaLabel.includes("Turn off microphone") || ariaLabel.includes("Disable microphone")) {
      return false;
    }
    const ariaPressed = element.getAttribute("aria-pressed");
    if (ariaPressed !== null) {
      return ariaPressed === "true";
    }
    const dataTooltip = element.getAttribute("data-tooltip") || "";
    if (dataTooltip.includes("unmute") || dataTooltip.includes("Turn on microphone")) {
      return true;
    }
    if (dataTooltip.includes("mute") || dataTooltip.includes("Turn off microphone")) {
      return false;
    }
    const hasMuteIcon = element.querySelector('[data-icon-name*="mic_off"], [data-icon-name*="mute"], .mic-off, .mute-icon');
    const hasUnmuteIcon = element.querySelector('[data-icon-name*="mic"], [data-icon-name*="unmute"], .mic-on, .unmute-icon');
    if (hasMuteIcon && !hasUnmuteIcon) {
      return true;
    }
    if (hasUnmuteIcon && !hasMuteIcon) {
      return false;
    }
    if (element.hasAttribute("disabled") || element.getAttribute("aria-disabled") === "true") {
      return true;
    }
    const classList = element.className.toLowerCase();
    if (classList.includes("muted") || classList.includes("mic-off")) {
      return true;
    }
    if (classList.includes("unmuted") || classList.includes("mic-on")) {
      return false;
    }
    return null;
  }
  setupObserver(muteButton) {
    if (this.observer) {
      this.observer.disconnect();
    }
    this.currentObservedElement = muteButton;
    this.observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        if (mutation.type === "attributes") {
          const target = mutation.target;
          let isMuted = false;
          if (target.hasAttribute("data-is-muted")) {
            isMuted = target.getAttribute("data-is-muted") === "true";
          } else if (target.getAttribute("aria-label")?.includes("unmute")) {
            isMuted = true;
          } else if (target.getAttribute("aria-label")?.includes("mute")) {
            isMuted = false;
          } else if (target.getAttribute("aria-pressed") === "true") {
            isMuted = true;
          } else if (target.getAttribute("aria-pressed") === "false") {
            isMuted = false;
          }
          if (this.lastKnownState !== isMuted) {
            this.lastKnownState = isMuted;
            this.updateActivity();
            chrome.runtime.sendMessage({
              type: "muteStateChanged",
              isMuted,
              timestamp: (/* @__PURE__ */ new Date()).toISOString(),
              source: "mutation-observer"
            });
          }
        }
      });
    });
    this.observer.observe(muteButton, {
      attributes: true,
      attributeFilter: ["data-is-muted", "aria-label", "aria-pressed"]
    });
  }
}
class ParticipantMonitor extends BaseMonitor {
  observer;
  participantCount;
  participants;
  constructor() {
    super();
    this.observer = null;
    this.participantCount = 0;
    this.participants = /* @__PURE__ */ new Map();
  }
  start() {
    this.startParticipantMonitoring();
  }
  stop() {
    if (this.observer) {
      this.observer.disconnect();
    }
  }
  startParticipantMonitoring() {
    const participantList = document.querySelector('[jsname="BOHaEe"]');
    if (participantList) {
      this.observer = new MutationObserver((mutations) => {
        let participantChanged = false;
        mutations.forEach((mutation) => {
          if (mutation.type === "childList") {
            participantChanged = true;
          }
        });
        if (participantChanged) {
          const participantList2 = document.querySelector('[jsname="BOHaEe"]');
          if (participantList2) {
            this.detectParticipants();
          }
        }
      });
      this.observer.observe(participantList, {
        childList: true,
        subtree: true
      });
      this.detectParticipants();
    }
  }
  detectParticipants() {
    const participantElements = document.querySelectorAll('[jsname="BOHaEe"] > div');
    const newParticipants = /* @__PURE__ */ new Map();
    let newCount = 0;
    participantElements.forEach((element, index) => {
      const nameElement = element.querySelector('[jsname="BOHaEe"] span');
      const name = nameElement ? nameElement.textContent || `Participant ${index + 1}` : `Participant ${index + 1}`;
      const participant = {
        id: `participant-${index}`,
        name,
        isHost: element.querySelector("[data-is-host]") !== null,
        isPresenter: element.querySelector("[data-is-presenter]") !== null,
        isScreenSharing: element.querySelector("[data-is-screen-sharing]") !== null,
        isVideoOn: element.querySelector("[data-is-video-on]") !== null,
        isAudioOn: element.querySelector("[data-is-audio-on]") !== null
      };
      newParticipants.set(participant.id, participant);
      newCount++;
    });
    this.participants = newParticipants;
    this.updateParticipantCount(newCount);
  }
  updateParticipantCount(count) {
    if (count !== this.participantCount) {
      this.participantCount = count;
      this.updateActivity();
      chrome.runtime.sendMessage({
        type: "participantCountChanged",
        count,
        timestamp: (/* @__PURE__ */ new Date()).toISOString()
      });
      this.notifyParticipantsUpdate();
      setTimeout(() => {
        const muteMonitor = window.muteStateMonitor;
        if (muteMonitor && typeof muteMonitor.checkAndNotifyMuteState === "function") {
          muteMonitor.checkAndNotifyMuteState();
        }
      }, 1e3);
    }
  }
  notifyParticipantsUpdate() {
    const participantsArray = Array.from(this.participants.values());
    this.updateActivity();
    chrome.runtime.sendMessage({
      type: "participantsUpdate",
      participants: participantsArray,
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    });
  }
}
class ScreenShareMonitor extends BaseMonitor {
  observer;
  isScreenSharing;
  constructor() {
    super();
    this.observer = null;
    this.isScreenSharing = false;
  }
  start() {
    this.startScreenShareMonitoring();
  }
  stop() {
    if (this.observer) {
      this.observer.disconnect();
    }
  }
  startScreenShareMonitoring() {
    const screenShareButton = document.querySelector('[jsname="BOHaEe"] [data-is-screen-sharing]');
    if (screenShareButton) {
      this.observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          if (mutation.type === "attributes" && mutation.attributeName === "data-is-screen-sharing") {
            const target = mutation.target;
            const isScreenSharing = target.getAttribute("data-is-screen-sharing") === "true";
            if (this.isScreenSharing !== isScreenSharing) {
              this.isScreenSharing = isScreenSharing;
              this.updateActivity();
              this.notifyScreenShareChange(isScreenSharing);
            }
          }
        });
      });
      this.observer.observe(screenShareButton, {
        attributes: true,
        attributeFilter: ["data-is-screen-sharing"]
      });
    }
  }
  notifyScreenShareChange(isScreenSharing) {
    this.updateActivity();
    chrome.runtime.sendMessage({
      type: "screenShareChanged",
      isScreenSharing,
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    });
  }
}
class ChatMonitor extends BaseMonitor {
  observer;
  constructor() {
    super();
    this.observer = null;
  }
  start() {
    this.startChatMonitoring();
  }
  stop() {
    if (this.observer) {
      this.observer.disconnect();
    }
  }
  startChatMonitoring() {
    const chatContainer = document.querySelector('[jsname="BOHaEe"]');
    if (chatContainer) {
      this.observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType === Node.ELEMENT_NODE) {
              const element = node;
              if (element.querySelector('[jsname="VfPpkd-LgbsSe"]')) {
                this.detectNewChatMessage(element);
              }
            }
          });
        });
      });
      this.observer.observe(chatContainer, {
        childList: true,
        subtree: true
      });
    }
  }
  detectNewChatMessage(chatElement) {
    const messageText = chatElement.textContent || "";
    if (messageText.trim()) {
      this.updateActivity();
      chrome.runtime.sendMessage({
        type: "newChatMessage",
        message: messageText,
        timestamp: (/* @__PURE__ */ new Date()).toISOString(),
        sender: "unknown"
      });
    }
  }
}
class LeaveCallButtonMonitor extends BaseMonitor {
  observer;
  constructor() {
    super();
    this.observer = null;
  }
  start() {
    this.startLeaveCallButtonMonitoring();
  }
  stop() {
    if (this.observer) {
      this.observer.disconnect();
    }
  }
  startLeaveCallButtonMonitoring() {
    this.setupLeaveCallButtonListeners();
    this.setupLeaveCallButtonObserver();
  }
  setupLeaveCallButtonListeners() {
    const leaveCallSelectors = [
      'button[aria-label*="Leave call"]',
      'button[aria-label*="End call"]',
      'button[data-tooltip*="Leave call"]',
      'button[data-tooltip*="End call"]',
      '[jsname="CQylAd"]',
      // Leave call button jsname
      'div[role="button"][aria-label*="Leave"]',
      'div[role="button"][aria-label*="End call"]',
      'button[aria-label*="Hang up"]',
      'button[aria-label*="Disconnect"]'
    ];
    leaveCallSelectors.forEach((selector) => {
      const buttons = document.querySelectorAll(selector);
      buttons.forEach((button) => {
        this.addLeaveCallClickListener(button);
      });
    });
  }
  setupLeaveCallButtonObserver() {
    this.observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const element = node;
            const leaveCallSelectors = [
              'button[aria-label*="Leave call"]',
              'button[aria-label*="End call"]',
              'button[data-tooltip*="Leave call"]',
              'button[data-tooltip*="End call"]',
              '[jsname="CQylAd"]',
              'div[role="button"][aria-label*="Leave"]',
              'div[role="button"][aria-label*="End call"]',
              'button[aria-label*="Hang up"]',
              'button[aria-label*="Disconnect"]'
            ];
            leaveCallSelectors.forEach((selector) => {
              if (element.matches && element.matches(selector)) {
                this.addLeaveCallClickListener(element);
              }
              const buttons = element.querySelectorAll(selector);
              buttons.forEach((button) => {
                this.addLeaveCallClickListener(button);
              });
            });
          }
        });
      });
    });
    this.observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }
  addLeaveCallClickListener(button) {
    if (button.hasAttribute("data-leave-call-listener-added")) {
      return;
    }
    const clickHandler = () => {
      this.updateActivity();
      this.triggerLeaveCallCleanup();
    };
    button.addEventListener("click", clickHandler);
    button.setAttribute("data-leave-call-listener-added", "true");
  }
  triggerLeaveCallCleanup() {
    const event = new CustomEvent("recordingStoppedByUser", {
      detail: {
        reason: "user_left_call",
        timestamp: (/* @__PURE__ */ new Date()).toISOString(),
        source: "leave_call_button"
      }
    });
    window.dispatchEvent(event);
    chrome.runtime.sendMessage({
      type: "recordingStoppedByUser",
      reason: "user_left_call",
      timestamp: (/* @__PURE__ */ new Date()).toISOString(),
      source: "leave_call_button"
    }).catch((_error) => {
    });
  }
}
class MeetingEndMonitor extends BaseMonitor {
  observer;
  checkInterval;
  hasNotifiedMeetingEnd;
  lastKnownUrl;
  constructor() {
    super();
    this.observer = null;
    this.checkInterval = null;
    this.hasNotifiedMeetingEnd = false;
    this.lastKnownUrl = window.location.href;
  }
  start() {
    this.startMeetingEndDetection();
    this.startUrlMonitoring();
  }
  stop() {
    if (this.observer) {
      this.observer.disconnect();
    }
    if (this.checkInterval) {
      clearInterval(this.checkInterval);
      this.checkInterval = null;
    }
  }
  startMeetingEndDetection() {
    this.setupMeetingEndObserver();
    this.checkInterval = window.setInterval(() => {
      this.checkForMeetingEnd();
    }, 2e3);
  }
  startUrlMonitoring() {
    setInterval(() => {
      const currentUrl = window.location.href;
      if (currentUrl !== this.lastKnownUrl) {
        this.lastKnownUrl = currentUrl;
        const oldMeetingId = this.extractMeetingId(this.lastKnownUrl);
        const newMeetingId = this.extractMeetingId(currentUrl);
        if (oldMeetingId && newMeetingId && oldMeetingId !== newMeetingId) {
          this.notifyMeetingEnd("meeting_id_changed");
        }
      }
    }, 1e3);
  }
  extractMeetingId(url) {
    const match = url.match(/meet\.google\.com\/([a-z]{3}-[a-z]{4}-[a-z]{3})/);
    return match ? match[1] : null;
  }
  setupMeetingEndObserver() {
    this.observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const elementNode = node;
            const text = elementNode.textContent || "";
            if (text.includes("Call ended") || text.includes("Meeting ended")) {
              this.notifyMeetingEnd("call_ended_detected");
            }
          }
        });
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const elementNode = node;
            if (elementNode.querySelector('[jsname="HJwP7e"]')) {
              this.notifyMeetingEnd("leave_button_detected");
            }
          }
        });
      });
    });
    this.observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }
  checkForMeetingEnd() {
    const indicators = this.getMeetingEndIndicators();
    if (indicators.hasMeetingEnded) {
      this.notifyMeetingEnd(indicators.reason);
    }
    if (document.hidden) {
      setTimeout(() => {
        if (document.hidden) {
          this.notifyMeetingEnd("page_hidden");
        }
      }, 5e3);
    }
  }
  getMeetingEndIndicators() {
    const thankYouMessages = [
      "Thank you for using Google Meet",
      "Call ended",
      "Meeting ended",
      "You left the meeting"
    ];
    for (const message of thankYouMessages) {
      if (document.body.textContent && document.body.textContent.includes(message)) {
        return { hasMeetingEnded: true, reason: "thank_you_message" };
      }
    }
    const preJoinScreen = document.querySelector('[jsname="BOHaEe"]');
    if (preJoinScreen && this.hasBeenInMeeting()) {
      return { hasMeetingEnded: true, reason: "pre_join_screen" };
    }
    const controls = document.querySelector('[jsname="BOHaEe"]');
    if (!controls && this.hasBeenInMeeting()) {
      return { hasMeetingEnded: true, reason: "no_controls" };
    }
    return { hasMeetingEnded: false, reason: "" };
  }
  hasBeenInMeeting() {
    return this.lastKnownUrl.includes("meet.google.com/");
  }
  notifyMeetingEnd(reason) {
    if (this.hasNotifiedMeetingEnd) {
      return;
    }
    this.hasNotifiedMeetingEnd = true;
    this.updateActivity();
    chrome.runtime.sendMessage({
      type: "meetingEnded",
      reason,
      timestamp: (/* @__PURE__ */ new Date()).toISOString(),
      url: window.location.href
    });
    chrome.runtime.sendMessage({
      type: "MEETING_ID_IN_CS",
      id: this.extractMeetingId(window.location.href)
    });
  }
}
class MeetMuteDetector {
  muteStateMonitor;
  participantMonitor;
  screenShareMonitor;
  chatMonitor;
  meetingEndMonitor;
  leaveCallButtonMonitor;
  meetingStartTime;
  meetingStateInterval;
  // Legacy properties for backward compatibility (will be removed in future)
  observer;
  meetingEndObserver;
  isObserving;
  isMeetingEndObserving;
  lastKnownState;
  lastKnownUrl;
  hasNotifiedMeetingEnd;
  participantCount;
  participants;
  constructor() {
    this.muteStateMonitor = new MuteStateMonitor();
    this.participantMonitor = new ParticipantMonitor();
    this.screenShareMonitor = new ScreenShareMonitor();
    this.chatMonitor = new ChatMonitor();
    this.meetingEndMonitor = new MeetingEndMonitor();
    this.leaveCallButtonMonitor = new LeaveCallButtonMonitor();
    this.meetingStartTime = null;
    this.meetingStateInterval = null;
    this.observer = null;
    this.meetingEndObserver = null;
    this.isObserving = false;
    this.isMeetingEndObserving = false;
    this.lastKnownState = null;
    this.lastKnownUrl = window.location.href;
    this.hasNotifiedMeetingEnd = false;
    this.participantCount = 0;
    this.participants = /* @__PURE__ */ new Map();
    this.setupMessageListener();
  }
  setupMessageListener() {
    chrome.runtime.onMessage.addListener((request, _sender, sendResponse) => {
      if (request.type === "getMuteState") {
        const muteButton = document.querySelector("[data-is-muted]");
        const isMuted = muteButton ? muteButton.getAttribute("data-is-muted") === "true" : null;
        sendResponse({ isMuted });
        return true;
      }
      if (request.type === "leaveMeeting") {
        this.leaveMeeting();
        sendResponse({ success: true });
        return true;
      }
      if (request.type === "getMeetingState") {
        const meetingState = this.getCurrentMeetingState();
        sendResponse(meetingState);
        return true;
      }
      if (request.type === "getParticipants") {
        sendResponse({ participants: [] });
        return true;
      }
      if (request.type === "toggleScreenShare") {
        this.toggleScreenShare();
        sendResponse({ success: true });
        return true;
      }
      if (request.type === "startRecording") {
        this.startMeetingRecording();
        sendResponse({ success: true });
        return true;
      }
      if (request.type === "stopRecording") {
        this.stopMeetingRecording();
        sendResponse({ success: true });
        return true;
      }
    });
  }
  start() {
    this.muteStateMonitor.start();
    this.participantMonitor.start();
    this.screenShareMonitor.start();
    this.chatMonitor.start();
    this.meetingEndMonitor.start();
    this.leaveCallButtonMonitor.start();
    this.startMeetingStateTracking();
  }
  // This method is now handled by the individual monitor classes
  // Participant monitoring is now handled by ParticipantMonitor class
  // Screen share monitoring is now handled by ScreenShareMonitor class
  // Chat monitoring is now handled by ChatMonitor class
  startMeetingControlsMonitoring() {
    const recordingIndicator = document.querySelector("[data-is-recording]") || document.querySelector('[aria-label*="recording"]') || document.querySelector('[jsname="recording"]');
    if (recordingIndicator) {
      this.notifyRecordingStateChange(true);
    }
  }
  startMeetingStateTracking() {
    this.meetingStateInterval = setInterval(() => {
      this.updateMeetingState();
    }, 5e3);
  }
  updateMeetingState() {
    const currentState = this.getCurrentMeetingState();
    chrome.runtime.sendMessage({
      type: "meetingStateUpdate",
      state: currentState
    }).catch((_error) => {
    });
  }
  getCurrentMeetingState() {
    const now = /* @__PURE__ */ new Date();
    const duration = this.meetingStartTime ? Math.floor((now.getTime() - this.meetingStartTime.getTime()) / 1e3) : 0;
    return {
      isInMeeting: this.hasBeenInMeeting(),
      participantCount: 0,
      // TODO: Get from participant monitor
      isScreenSharing: false,
      // TODO: Get from screen share monitor
      isRecording: this.checkRecordingState(),
      meetingDuration: duration,
      lastActivity: /* @__PURE__ */ new Date()
      // TODO: Get from monitors
    };
  }
  updateParticipantCount() {
    const participantSelectors = [
      "[data-participant-id]",
      '[jsname="VfPpkd-LgbsSe"]',
      '[aria-label*="participant"]',
      'div[role="listitem"]'
    ];
    let count = 0;
    participantSelectors.forEach((selector) => {
      const elements = document.querySelectorAll(selector);
      count = Math.max(count, elements.length);
    });
    if (count !== this.participantCount) {
      this.participantCount = count;
      this.notifyParticipantCountChange(count);
    }
  }
  detectParticipants() {
    const participantElements = document.querySelectorAll('[data-participant-id], [jsname="VfPpkd-LgbsSe"]');
    participantElements.forEach((element, index) => {
      const participantId = element.getAttribute("data-participant-id") || `participant_${index}`;
      const name = element.getAttribute("aria-label") || element.textContent || `Participant ${index + 1}`;
      const isHost = element.querySelector('[aria-label*="host"]') !== null;
      const isPresenter = element.querySelector('[aria-label*="presenter"]') !== null;
      const isScreenSharing = element.querySelector('[aria-label*="screen"]') !== null;
      const isVideoOn = element.querySelector('[aria-label*="camera off"]') === null;
      const isAudioOn = element.querySelector('[aria-label*="microphone off"]') === null;
      const participantInfo = {
        id: participantId,
        name: name.trim(),
        isHost,
        isPresenter,
        isScreenSharing,
        isVideoOn,
        isAudioOn
      };
      this.participants.set(participantId, participantInfo);
    });
    this.notifyParticipantsUpdate();
  }
  checkScreenSharingState() {
    const screenShareIndicators = [
      '[data-is-presenting="true"]',
      '[aria-label*="presenting"]',
      '[jsname="presenting"]',
      'div[aria-label*="screen"]'
    ];
    return screenShareIndicators.some(
      (selector) => document.querySelector(selector) !== null
    );
  }
  checkRecordingState() {
    const recordingIndicators = [
      '[data-is-recording="true"]',
      '[aria-label*="recording"]',
      '[jsname="recording"]',
      'div[aria-label*="recording"]'
    ];
    return recordingIndicators.some(
      (selector) => document.querySelector(selector) !== null
    );
  }
  detectNewChatMessage(chatElement) {
    const messageText = chatElement.textContent || "";
    const timestamp = (/* @__PURE__ */ new Date()).toISOString();
    if (messageText.trim()) {
      chrome.runtime.sendMessage({
        type: "newChatMessage",
        message: messageText.trim(),
        timestamp,
        sender: "unknown"
      }).catch((_error) => {
      });
    }
  }
  notifyScreenShareChange(isScreenSharing) {
    chrome.runtime.sendMessage({
      type: "screenShareChanged",
      isScreenSharing,
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    }).catch((_error) => {
    });
  }
  notifyParticipantCountChange(count) {
    chrome.runtime.sendMessage({
      type: "participantCountChanged",
      count,
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    }).catch((_error) => {
    });
  }
  notifyParticipantsUpdate() {
    const participants = Array.from(this.participants.values());
    chrome.runtime.sendMessage({
      type: "participantsUpdate",
      participants,
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    }).catch((_error) => {
    });
  }
  notifyRecordingStateChange(isRecording) {
    chrome.runtime.sendMessage({
      type: "recordingStateChanged",
      isRecording,
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    }).catch((_error) => {
    });
  }
  toggleScreenShare() {
    const screenShareButton = document.querySelector('[aria-label*="present"]') || document.querySelector('[jsname="present"]') || document.querySelector('button[aria-label*="screen"]');
    if (screenShareButton) {
      screenShareButton.click();
    }
  }
  startMeetingRecording() {
    const recordButton = document.querySelector('[aria-label*="record"]') || document.querySelector('[jsname="record"]') || document.querySelector('button[aria-label*="recording"]');
    if (recordButton) {
      recordButton.click();
    }
  }
  stopMeetingRecording() {
    const stopRecordButton = document.querySelector('[aria-label*="stop recording"]') || document.querySelector('[jsname="stopRecord"]') || document.querySelector('button[aria-label*="stop recording"]');
    if (stopRecordButton) {
      stopRecordButton.click();
    }
  }
  startMeetingEndDetection() {
    if (this.isMeetingEndObserving) return;
    this.isMeetingEndObserving = true;
    setInterval(() => {
      this.checkForMeetingEnd();
    }, 2e3);
    this.startUrlMonitoring();
    this.setupMeetingEndObserver();
    window.addEventListener("beforeunload", () => {
      this.notifyMeetingEnd("page_unload");
    });
    document.addEventListener("visibilitychange", () => {
      if (document.hidden) {
        setTimeout(() => {
          if (document.hidden) {
            this.checkForMeetingEnd();
          }
        }, 5e3);
      }
    });
  }
  checkForMeetingEnd() {
    try {
      const indicators = this.getMeetingEndIndicators();
      if (indicators.hasMeetingEnded) {
        this.notifyMeetingEnd(indicators.reason);
        return;
      }
      if (!window.location.href.includes("meet.google.com") || !window.location.href.match(/meet\.google\.com\/[a-z\-]+/)) {
        this.notifyMeetingEnd("left_meet_page");
        return;
      }
    } catch (error) {
    }
  }
  getMeetingEndIndicators() {
    const indicators = {
      hasMeetingEnded: false,
      reason: ""
    };
    const thankYouMessages = [
      "Thanks for joining",
      "You left the meeting",
      "Meeting ended",
      "Call ended",
      "You have left the call"
    ];
    for (const message of thankYouMessages) {
      if (document.body.textContent && document.body.textContent.includes(message)) {
        indicators.hasMeetingEnded = true;
        indicators.reason = "thank_you_message";
        return indicators;
      }
    }
    const preJoinScreen = document.querySelector("[data-prewarm-step]") || document.querySelector('div[aria-label*="Join"]') || document.querySelector('button[aria-label*="Join now"]');
    if (preJoinScreen && this.hasBeenInMeeting()) {
      indicators.hasMeetingEnded = true;
      indicators.reason = "back_to_prejoin";
      return indicators;
    }
    const controls = document.querySelector("[data-is-muted]") || document.querySelector('[aria-label*="microphone"]') || document.querySelector('[aria-label*="camera"]');
    if (!controls && this.hasBeenInMeeting()) {
      indicators.hasMeetingEnded = true;
      indicators.reason = "controls_gone";
      return indicators;
    }
    return indicators;
  }
  hasBeenInMeeting() {
    return this.lastKnownState !== null;
  }
  startUrlMonitoring() {
    setInterval(() => {
      const currentUrl = window.location.href;
      if (currentUrl !== this.lastKnownUrl) {
        if (this.lastKnownUrl.includes("meet.google.com") && !currentUrl.includes("meet.google.com")) {
          this.notifyMeetingEnd("navigated_away");
        }
        const oldMeetingId = this.extractMeetingId(this.lastKnownUrl);
        const newMeetingId = this.extractMeetingId(currentUrl);
        if (oldMeetingId && newMeetingId && oldMeetingId !== newMeetingId) {
          this.notifyMeetingEnd("meeting_id_changed");
        }
        this.lastKnownUrl = currentUrl;
      }
    }, 1e3);
  }
  extractMeetingId(url) {
    const match = url.match(/https:\/\/meet\.google\.com\/([a-z]{3}-[a-z]{4}-[a-z]{3})/);
    return match ? match[1] : null;
  }
  setupMeetingEndObserver() {
    this.meetingEndObserver = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const text = node.textContent || "";
            if (text.includes("Thanks for joining") || text.includes("You left the meeting") || text.includes("Call ended")) {
              this.notifyMeetingEnd("end_message_appeared");
            }
          }
        });
        mutation.removedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const elementNode = node;
            if (elementNode.querySelector && (elementNode.querySelector("[data-meeting-id]") || elementNode.querySelector('[jsname="HJwP7e"]'))) {
              this.notifyMeetingEnd("meeting_ui_removed");
            }
          }
        });
      });
    });
    this.meetingEndObserver.observe(document.body, {
      childList: true,
      subtree: true
    });
  }
  notifyMeetingEnd(reason) {
    if (this.hasNotifiedMeetingEnd) {
      return;
    }
    this.hasNotifiedMeetingEnd = true;
    chrome.runtime.sendMessage({
      type: "meetingEnded",
      reason,
      timestamp: (/* @__PURE__ */ new Date()).toISOString(),
      url: window.location.href
    }).catch((_error) => {
    });
    setTimeout(() => {
      this.hasNotifiedMeetingEnd = false;
    }, 3e4);
  }
  leaveMeeting() {
    try {
      const leaveButtons = [
        'button[aria-label*="Leave call"]',
        'button[aria-label*="End call"]',
        'button[data-tooltip*="Leave call"]',
        'button[data-tooltip*="End call"]',
        '[jsname="CQylAd"]',
        // Leave call button jsname
        'div[role="button"][aria-label*="Leave"]'
      ];
      for (const selector of leaveButtons) {
        const button = document.querySelector(selector);
        if (button) {
          button.click();
          return true;
        }
      }
      document.dispatchEvent(new KeyboardEvent("keydown", {
        key: "d",
        code: "KeyD",
        ctrlKey: true,
        bubbles: true
      }));
      setTimeout(() => {
        window.location.href = "https://meet.google.com/";
      }, 2e3);
      return true;
    } catch (error) {
      return false;
    }
  }
  checkAndNotifyMuteState() {
    const muteButton = document.querySelector("[data-is-muted]");
    if (muteButton) {
      const isMuted = muteButton.getAttribute("data-is-muted") === "true";
      if (this.lastKnownState !== isMuted) {
        this.lastKnownState = isMuted;
        chrome.runtime.sendMessage({
          type: "muteStateChanged",
          isMuted
        });
      }
      if (!this.isObserving) {
        this.setupObserver(muteButton);
        this.isObserving = true;
      }
    }
  }
  setupObserver(muteButton) {
    this.observer = new MutationObserver((mutations) => {
      mutations.forEach((mutation) => {
        if (mutation.type === "attributes" && mutation.attributeName === "data-is-muted") {
          const target = mutation.target;
          const isMuted = target.getAttribute("data-is-muted") === "true";
          if (this.lastKnownState !== isMuted) {
            this.lastKnownState = isMuted;
            chrome.runtime.sendMessage({
              type: "muteStateChanged",
              isMuted
            });
          }
        }
      });
    });
    this.observer.observe(muteButton, {
      attributes: true,
      attributeFilter: ["data-is-muted"]
    });
  }
  stop() {
    this.muteStateMonitor.stop();
    this.participantMonitor.stop();
    this.screenShareMonitor.stop();
    this.chatMonitor.stop();
    this.meetingEndMonitor.stop();
    this.leaveCallButtonMonitor.stop();
    if (this.meetingStateInterval) {
      clearInterval(this.meetingStateInterval);
      this.meetingStateInterval = null;
    }
  }
  // Public getter for mute state
  getMuteState() {
    return this.lastKnownState;
  }
  // Public setter for mute state
  setMuteState(isMuted) {
    this.lastKnownState = isMuted;
  }
}
const detector = new MeetMuteDetector();
detector.start();
window.muteStateMonitor = detector.muteStateMonitor;
window.debugMuteDetection = () => {
  const muteSelectors = [
    "[data-is-muted]",
    '[aria-label*="microphone"]',
    '[jsname="BOHaEe"] button[aria-label*="microphone"]',
    'button[aria-label*="microphone"]',
    '[data-tooltip*="microphone"]',
    'button[data-tooltip*="microphone"]'
  ];
  muteSelectors.forEach((selector) => {
    document.querySelector(selector);
  });
};
window.forceMuteState = (isMuted) => {
  chrome.runtime.sendMessage({
    type: "muteStateChanged",
    isMuted,
    timestamp: (/* @__PURE__ */ new Date()).toISOString(),
    source: "manual-override"
  }).catch((_error) => {
  });
  try {
    window.dispatchEvent(new CustomEvent("muteStateChanged", {
      detail: { isMuted, timestamp: (/* @__PURE__ */ new Date()).toISOString() }
    }));
  } catch (error) {
  }
};
window.getCurrentMuteState = () => {
  return detector.getMuteState();
};
window.forceMuteStateUpdate = (isMuted) => {
  detector.setMuteState(isMuted);
  chrome.runtime.sendMessage({
    type: "muteStateChanged",
    isMuted,
    timestamp: (/* @__PURE__ */ new Date()).toISOString(),
    source: "manual-force"
  }).catch((_error) => {
  });
  try {
    window.dispatchEvent(new CustomEvent("muteStateChanged", {
      detail: { isMuted, timestamp: (/* @__PURE__ */ new Date()).toISOString() }
    }));
  } catch (error) {
  }
};
window.testMuteState = () => {
  const possibleMuteButtons = document.querySelectorAll('button[aria-label*="microphone"], button[aria-label*="mute"], button[data-tooltip*="microphone"]');
  possibleMuteButtons.forEach((button, _index) => {
    const ariaLabel = button.getAttribute("aria-label") || "";
    ariaLabel.includes("unmute") || ariaLabel.includes("Turn on microphone");
  });
};
chrome.runtime.sendMessage({
  type: "contentScriptReady"
});
(function() {
  const url = window.location.href;
  const match = url.match(/https:\/\/meet\.google\.com\/([a-z]{3}-[a-z]{4}-[a-z]{3})/);
  if (match) {
    const meetingId = match[1];
    chrome.runtime.sendMessage({
      type: "MEETING_ID_IN_CS",
      id: meetingId
    });
  }
})();
//# sourceMappingURL=meetDetector.ts-D6O_xnPS.js.map
})()
