import { A as AuthService, a as ApiEndpoints, u as useGlassTransparent, U as UrlConfig, j as jsxRuntimeExports, G as GlassTransparentProvider, c as clientExports } from "./index-BDbae9AU.js";
import { r as reactExports, R as React, T as Trash2, U as Upload, P as Play, a as Pause, S as Settings, C as Clock, b as Square } from "./lucide-CC3hCUHs.js";
import "./react-vendor-BS-dYsv0.js";
const useAuth = () => {
  const [user, setUser] = reactExports.useState(null);
  const [loading, setLoading] = reactExports.useState(true);
  const [error, setError] = reactExports.useState(null);
  const authService = AuthService.getInstance();
  reactExports.useEffect(() => {
    checkAuthStatus();
  }, []);
  const checkAuthStatus = async () => {
    try {
      setLoading(true);
      const currentUser = await authService.getCurrentUser();
      setUser(currentUser);
    } catch (err) {
      setError(err instanceof Error ? err.message : "Auth check failed");
    } finally {
      setLoading(false);
    }
  };
  const signIn = async () => {
    try {
      setLoading(true);
      setError(null);
      const userData = await authService.signIn();
      setUser(userData);
      return userData;
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : "Sign in failed";
      setError(errorMessage);
      throw new Error(errorMessage);
    } finally {
      setLoading(false);
    }
  };
  const signOut = async () => {
    try {
      setLoading(true);
      await authService.signOut();
      setUser(null);
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : "Sign out failed";
      setError(errorMessage);
      throw new Error(errorMessage);
    } finally {
      setLoading(false);
    }
  };
  const isUserAuthenticated = () => {
    return user !== null;
  };
  return {
    user,
    loading,
    error,
    signIn,
    signOut,
    isSignedIn: !!user,
    isUserAuthenticated,
    checkAuthStatus
  };
};
class MeetingService {
  static instance;
  _isMeetingInfoFound = false;
  currentMeetingIdGlobal = null;
  static getInstance() {
    if (!MeetingService.instance) {
      MeetingService.instance = new MeetingService();
    }
    return MeetingService.instance;
  }
  // Enhanced collectMeetingId function from working JavaScript code
  async collectMeetingId() {
    try {
      if (this.currentMeetingIdGlobal != null) {
        return this.currentMeetingIdGlobal;
      }
      let currentMeetingId = null;
      try {
        const response = await new Promise((resolve, reject) => {
          const timeout = setTimeout(() => {
            reject(new Error("Background script timeout"));
          }, 3e3);
          chrome.runtime.sendMessage({ type: "MEETING_ID_IN_BG" }, (response2) => {
            clearTimeout(timeout);
            if (chrome.runtime.lastError) {
              reject(new Error(chrome.runtime.lastError.message));
            } else {
              resolve(response2 || {});
            }
          });
        });
        currentMeetingId = response?.meetingId || "";
        if (currentMeetingId && currentMeetingId.trim()) {
          this.currentMeetingIdGlobal = currentMeetingId.trim();
          if (!this._isMeetingInfoFound) {
            await this.loadMeetingInfo();
          }
          return this.currentMeetingIdGlobal;
        }
      } catch (bgError) {
      }
      try {
        const { tabUrl } = await new Promise((resolve, reject) => {
          const timeout = setTimeout(() => {
            reject(new Error("Tab info timeout"));
          }, 2e3);
          chrome.runtime.sendMessage({ action: "checkTab" }, (response) => {
            clearTimeout(timeout);
            if (chrome.runtime.lastError) {
              reject(new Error(chrome.runtime.lastError.message));
            } else {
              resolve(response || {});
            }
          });
        });
        if (tabUrl) {
          const patterns = [
            /https:\/\/meet\.google\.com\/([a-z\-]{3,15})/i,
            // abc-def-ghi format
            /meet\.google\.com\/([a-zA-Z\-]{10,})/i,
            // general format
            /meet\.google\.com\/([a-z\-]+)/i
            // original format
          ];
          let matchGoogleMeetUrl = null;
          for (const pattern of patterns) {
            matchGoogleMeetUrl = tabUrl.match(pattern);
            if (matchGoogleMeetUrl) break;
          }
          currentMeetingId = matchGoogleMeetUrl ? matchGoogleMeetUrl[1] : null;
          if (currentMeetingId) {
            this.currentMeetingIdGlobal = currentMeetingId;
            return this.currentMeetingIdGlobal;
          }
        }
      } catch (tabError) {
      }
      try {
        if (typeof window !== "undefined" && window.location && window.location.href) {
          const currentUrl = window.location.href;
          const patterns = [
            /https:\/\/meet\.google\.com\/([a-z\-]{3,15})/i,
            // abc-def-ghi format
            /meet\.google\.com\/([a-zA-Z\-]{10,})/i,
            // general format
            /meet\.google\.com\/([a-z\-]+)/i
            // original format
          ];
          let meetUrlMatch = null;
          for (const pattern of patterns) {
            meetUrlMatch = currentUrl.match(pattern);
            if (meetUrlMatch) break;
          }
          if (meetUrlMatch && meetUrlMatch[1]) {
            this.currentMeetingIdGlobal = meetUrlMatch[1];
            return this.currentMeetingIdGlobal;
          }
        }
      } catch (windowError) {
      }
      try {
        const meetingData = await new Promise((resolve) => {
          chrome.storage.local.get(["currentMeetingId"], (result) => {
            resolve(result);
          });
        });
        if (meetingData?.currentMeetingId) {
          this.currentMeetingIdGlobal = meetingData.currentMeetingId;
          return this.currentMeetingIdGlobal;
        }
      } catch (storageError) {
      }
      return null;
    } catch (error) {
      return null;
    }
  }
  // Load meeting info - collects meeting ID from current Google Meet session
  async loadMeetingInfo(_userEmail) {
    await this.collectMeetingId();
    this._isMeetingInfoFound = true;
  }
  // Get meeting info found status
  isMeetingInfoFound() {
    return this._isMeetingInfoFound;
  }
  // Get current meeting ID
  getCurrentMeetingId() {
    return this.currentMeetingIdGlobal;
  }
  // Reset meeting service state for new session
  resetMeetingState() {
    this.currentMeetingIdGlobal = null;
    this._isMeetingInfoFound = false;
  }
}
class Logger {
  prefix;
  constructor(prefix) {
    this.prefix = prefix;
  }
  formatMessage(level, message, _data) {
    const timestamp = (/* @__PURE__ */ new Date()).toISOString();
    return `[${timestamp}] [${level}] [${this.prefix}] ${message}`;
  }
  debug(message, data) {
    {
      console.log(this.formatMessage("DEBUG", message), data || "");
    }
  }
  info(message, data) {
    {
      console.log(this.formatMessage("INFO", message), data || "");
    }
  }
  warn(message, data) {
    console.warn(this.formatMessage("WARN", message), data || "");
  }
  error(message, error) {
    console.error(this.formatMessage("ERROR", message), error || "");
  }
}
const logger = {
  app: new Logger("App"),
  meeting: new Logger("MeetingCoach"),
  websocket: new Logger("WebSocketAudioService"),
  auth: new Logger("AuthService"),
  service: new Logger("Service"),
  hook: new Logger("Hook"),
  component: new Logger("Component")
};
function downsampleTo16kHz(input, inputSampleRate) {
  if (inputSampleRate === 16e3) {
    const int16 = new Int16Array(input.length);
    for (let i = 0; i < input.length; i++) {
      int16[i] = Math.max(-1, Math.min(1, input[i])) * 32767;
    }
    return int16;
  }
  const sampleRateRatio = inputSampleRate / 16e3;
  const newLength = Math.round(input.length / sampleRateRatio);
  const result = new Int16Array(newLength);
  let offsetResult = 0;
  let offsetBuffer = 0;
  while (offsetResult < result.length) {
    const nextOffsetBuffer = Math.round((offsetResult + 1) * sampleRateRatio);
    let accum = 0;
    let count = 0;
    for (let i = offsetBuffer; i < nextOffsetBuffer && i < input.length; i++) {
      accum += input[i];
      count++;
    }
    result[offsetResult] = Math.max(-1, Math.min(1, accum / count)) * 32767;
    offsetResult++;
    offsetBuffer = nextOffsetBuffer;
  }
  return result;
}
class WebSocketAudioService {
  static instance;
  // WebSocket & Audio State
  ws = null;
  audioContext = null;
  processor = null;
  micSource = null;
  merger = null;
  elemSources = [];
  mediaStream = null;
  observer = null;
  micStateObserver = null;
  lastLevelLog = 0;
  // Recording State
  recording = false;
  paused = false;
  recordingStartTime = null;
  isMeetMicMuted = null;
  // Service instances
  meetingService;
  constructor() {
    this.meetingService = MeetingService.getInstance();
  }
  static getInstance() {
    if (!WebSocketAudioService.instance) {
      WebSocketAudioService.instance = new WebSocketAudioService();
    }
    return WebSocketAudioService.instance;
  }
  // ============================================
  // GOOGLE MEET MIC STATE DETECTION
  // ============================================
  /** Detects if the Google Meet microphone is muted */
  detectMeetMicState() {
    const micButton = document.querySelector('[data-is-muted="true"], [data-is-muted="false"]');
    if (micButton) {
      return micButton.getAttribute("data-is-muted") === "true";
    }
    const micButtonByAria = document.querySelector('[aria-label*="Turn on microphone"], [aria-label*="Turn off microphone"], [aria-label*="Mute microphone"], [aria-label*="Unmute microphone"]');
    if (micButtonByAria) {
      const ariaLabel = micButtonByAria.getAttribute("aria-label") || "";
      return ariaLabel.includes("Turn on microphone") || ariaLabel.includes("Unmute microphone");
    }
    const micButtonByClass = document.querySelector(".VfPpkd-Bz112c-LgbsSe[data-is-muted], .U26fgb[data-is-muted]");
    if (micButtonByClass) {
      return micButtonByClass.getAttribute("data-is-muted") === "true";
    }
    const micIcon = document.querySelector('svg[data-icon-name="mic-off"], svg[data-icon-name="mic-on"]');
    if (micIcon) {
      const iconName = micIcon.getAttribute("data-icon-name");
      return iconName === "mic-off";
    }
    return null;
  }
  /** Monitors Google Meet microphone state changes */
  startMicStateMonitoring() {
    const initialState = this.detectMeetMicState();
    this.isMeetMicMuted = initialState;
    const observer = new MutationObserver(() => {
      const currentState = this.detectMeetMicState();
      if (currentState !== null && currentState !== this.isMeetMicMuted) {
        this.isMeetMicMuted = currentState;
      }
    });
    observer.observe(document.body, {
      childList: true,
      subtree: true,
      attributes: true,
      attributeFilter: ["data-is-muted", "aria-label", "data-icon-name"]
    });
    this.micStateObserver = observer;
  }
  // ============================================
  // PARTICIPANT DETECTION
  // ============================================
  /** Detects all participants with audio in the meeting */
  detectParticipants() {
    const participants = [];
    const elements = Array.from(document.querySelectorAll("video, audio"));
    elements.forEach((el, idx) => {
      try {
        const stream = el.srcObject || (el.captureStream ? el.captureStream() : null);
        const hasAudio = !!(stream && stream.getAudioTracks().length > 0);
        if (!hasAudio) return;
        let participantName = `Participant ${idx + 1}`;
        const parent = el.closest("[data-participant-id], [data-requested-participant-id], [aria-label]");
        if (parent) {
          const ariaLabel = parent.getAttribute("aria-label");
          if (ariaLabel && ariaLabel.trim()) {
            participantName = ariaLabel.split(",")[0].trim() || participantName;
          }
        }
        if (participantName.startsWith("Participant")) {
          const parentDiv = el.closest("div[data-participant-id], div[data-requested-participant-id]");
          if (parentDiv) {
            const nameElement = parentDiv.querySelector("[data-self-name], [aria-label]");
            if (nameElement) {
              const name = nameElement.textContent?.trim() || nameElement.getAttribute("aria-label");
              if (name && name.length > 0 && name.length < 50) {
                participantName = name;
              }
            }
          }
        }
        const id = `participant-${idx}-${Date.now()}`;
        participants.push({ id, label: participantName, element: el });
      } catch (err) {
      }
    });
    return participants;
  }
  // ============================================
  // CLEANUP FUNCTIONS
  // ============================================
  cleanupAudio() {
    try {
      this.processor?.disconnect();
    } catch {
    }
    try {
      this.micSource?.disconnect();
    } catch {
    }
    try {
      this.merger?.disconnect();
    } catch {
    }
    try {
      this.elemSources.forEach((item) => item.node.disconnect());
    } catch {
    }
    try {
      this.audioContext?.close();
    } catch {
    }
    try {
      this.mediaStream?.getTracks().forEach((t) => t.stop());
    } catch {
    }
    this.processor = null;
    this.micSource = null;
    this.merger = null;
    this.elemSources = [];
    this.audioContext = null;
    this.mediaStream = null;
    try {
      this.observer?.disconnect();
    } catch {
    }
    this.observer = null;
    try {
      this.micStateObserver?.disconnect();
    } catch {
    }
    this.micStateObserver = null;
  }
  cleanupSocket() {
    try {
      this.ws?.close();
    } catch {
    }
    this.ws = null;
  }
  // ============================================
  // START TRANSCRIPTION
  // ============================================
  async startCapture(userId) {
    if (this.recording) return;
    try {
      if (!userId) {
        throw new Error("User ID is required to start transcription");
      }
      const meetingId = await this.getCurrentMeetingId();
      const participants = this.detectParticipants();
      const channelCount = 1 + participants.length;
      const params = new URLSearchParams({
        sample_rate: String(16e3),
        channels: String(channelCount),
        format_turns: String(true),
        speaker_0: "You",
        // Your microphone
        end_of_turn_confidence_threshold: String(0.7),
        min_end_of_turn_silence_when_confident: String(160),
        max_turn_silence: String(2400),
        meeting_id: meetingId || "unknown-meeting",
        user_id: String(userId)
      });
      participants.forEach((participant, idx) => {
        params.append(`speaker_${idx + 1}`, participant.label);
      });
      const url = `${"wss://sales-hawk-transcription-backend-service-dev-68718154471.asia-south1.run.app"}/ws/stream?${params.toString()}`;
      const ws = new WebSocket(url);
      this.ws = ws;
      ws.onopen = async () => {
        const media = await navigator.mediaDevices.getUserMedia({ audio: true });
        this.mediaStream = media;
        const audioContext = new (window.AudioContext || window.webkitAudioContext)();
        this.audioContext = audioContext;
        const micSource = audioContext.createMediaStreamSource(media);
        this.micSource = micSource;
        const currentParticipants = this.detectParticipants();
        const totalChannels = 1 + currentParticipants.length;
        const merger = audioContext.createChannelMerger(totalChannels);
        this.merger = merger;
        micSource.connect(merger, 0, 0);
        currentParticipants.forEach((participant, idx) => {
          try {
            const stream = participant.element.srcObject || (participant.element.captureStream ? participant.element.captureStream() : null);
            if (stream) {
              const node = audioContext.createMediaStreamSource(stream);
              const channelIndex = idx + 1;
              node.connect(merger, 0, channelIndex);
              this.elemSources.push({
                node,
                id: participant.id,
                label: participant.label,
                channelIndex
              });
            } else {
              const node = audioContext.createMediaElementSource(participant.element);
              const channelIndex = idx + 1;
              node.connect(merger, 0, channelIndex);
              this.elemSources.push({
                node,
                id: participant.id,
                label: participant.label,
                channelIndex
              });
            }
          } catch (err) {
          }
        });
        const attachMeetElements = () => {
          if (!this.audioContext || !this.merger) return;
          const seen = new Set(this.elemSources.map((item) => item.id));
          const elements = Array.from(document.querySelectorAll("video, audio"));
          elements.forEach((el, idx) => {
            try {
              const stream = el.srcObject || (el.captureStream ? el.captureStream() : null);
              const hasAudio = !!(stream && stream.getAudioTracks().length > 0);
              if (!hasAudio) return;
              const id = `dynamic-${idx}-${Date.now()}`;
              if (seen.has(id)) return;
              let participantName = `Participant ${this.elemSources.length + 1}`;
              const parent = el.closest("[aria-label]");
              if (parent) {
                const ariaLabel = parent.getAttribute("aria-label");
                if (ariaLabel && ariaLabel.trim()) {
                  participantName = ariaLabel.split(",")[0].trim() || participantName;
                }
              }
            } catch {
            }
          });
        };
        attachMeetElements();
        try {
          const obs = new MutationObserver(() => attachMeetElements());
          this.observer = obs;
          obs.observe(document.body, { childList: true, subtree: true });
        } catch {
        }
        const processor = audioContext.createScriptProcessor(4096, totalChannels, 1);
        this.processor = processor;
        processor.onaudioprocess = (e) => {
          if (this.paused) return;
          const inputBuffer = e.inputBuffer;
          const numChannels = inputBuffer.numberOfChannels;
          if (numChannels < 1) return;
          const micMuted = this.isMeetMicMuted === true;
          const channels = [];
          for (let i = 0; i < numChannels; i++) {
            const channelData = inputBuffer.getChannelData(i);
            const processedChannel = new Float32Array(channelData.length);
            if (i === 0 && micMuted) {
              processedChannel.fill(0);
            } else {
              processedChannel.set(channelData);
            }
            channels.push(processedChannel);
          }
          const now = Date.now();
          if (now - this.lastLevelLog > 2e3) {
            this.lastLevelLog = now;
          }
          const downsampledChannels = channels.map((ch) => downsampleTo16kHz(ch, audioContext.sampleRate));
          const samplesPerChannel = downsampledChannels[0].length;
          const interleavedData = new Int16Array(samplesPerChannel * numChannels);
          for (let sample = 0; sample < samplesPerChannel; sample++) {
            for (let ch = 0; ch < numChannels; ch++) {
              interleavedData[sample * numChannels + ch] = downsampledChannels[ch][sample];
            }
          }
          if (ws.readyState === WebSocket.OPEN) {
            ws.send(interleavedData.buffer);
          }
        };
        merger.connect(processor);
        processor.connect(audioContext.destination);
        this.startMicStateMonitoring();
        this.recording = true;
        this.recordingStartTime = Date.now();
      };
      ws.onmessage = (event) => {
        try {
          const msg = JSON.parse(event.data);
          const type = msg?.type;
          logger.websocket.debug("Received message", { type, msg });
          if (type === "SessionBegins") {
            logger.websocket.info("Session started", msg.session_id);
            return;
          }
          if (type === "PartialTranscript") {
            logger.websocket.debug("PartialTranscript", msg.text);
            if (msg.text && msg.text.trim()) {
              const speakerName = msg.speaker || `Channel ${msg.channel ?? 0}`;
              const speakerType = speakerName === "You" ? "me" : "other";
              logger.websocket.debug("Emitting partial transcript", { speakerName, text: msg.text });
              this.emitPartialTranscriptEvent(speakerType, msg.text, speakerName);
            }
            return;
          }
          if (type === "Turn") {
            const transcript = msg.transcript || "";
            const formatted = msg.turn_is_formatted;
            logger.websocket.debug("Turn", { transcript, formatted, speaker: msg.speaker });
            if (!formatted && transcript && transcript.trim()) {
              const speakerName = msg.speaker || `Channel ${msg.channel ?? 0}`;
              const speakerType = speakerName === "You" ? "me" : "other";
              logger.websocket.debug("Emitting unformatted turn as partial", { speakerName, transcript });
              this.emitPartialTranscriptEvent(speakerType, transcript, speakerName);
            }
            if (formatted && transcript && transcript.trim()) {
              const speakerName = msg.speaker || `Channel ${msg.channel ?? 0}`;
              const speakerType = speakerName === "You" ? "me" : "other";
              const transcriptId = String(msg.turn_id || `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`);
              logger.websocket.info("Emitting final transcript", { speakerName, transcript });
              this.emitTranscriptionEvent(speakerType, transcript, transcriptId, speakerName);
            }
            return;
          }
          if (type === "FinalTranscript") {
            const transcript = msg.text || "";
            if (transcript && transcript.trim()) {
              const speakerName = msg.speaker || `Channel ${msg.channel ?? 0}`;
              const speakerType = speakerName === "You" ? "me" : "other";
              const transcriptId = String(`${Date.now()}-${Math.random().toString(36).substr(2, 9)}`);
              this.emitTranscriptionEvent(speakerType, transcript, transcriptId, speakerName);
            }
            return;
          }
          if (type === "SessionTerminated" || type === "Termination") {
            return;
          }
          if (type === "Error") {
            return;
          }
        } catch (err) {
        }
      };
      ws.onerror = (_error) => {
        this.stopCapture();
      };
      ws.onclose = () => {
        this.recording = false;
        this.cleanupAudio();
      };
    } catch (error) {
      throw error;
    }
  }
  // ============================================
  // STOP TRANSCRIPTION
  // ============================================
  async stopCapture() {
    if (!this.recording) return;
    try {
      if (this.ws && this.ws.readyState === WebSocket.OPEN) {
        this.ws.send(JSON.stringify({ type: "Terminate" }));
      }
      this.cleanupSocket();
      this.cleanupAudio();
      this.recording = false;
      this.recordingStartTime = null;
    } catch (error) {
      throw error;
    }
  }
  // ============================================
  // PAUSE/RESUME - Pauses audio stream only, keeps transcription session active
  // ============================================
  async pauseCapture() {
    if (!this.recording || this.paused) return;
    try {
      this.paused = true;
      logger.websocket.info("Audio capture paused - transcription session remains active");
    } catch (error) {
      logger.websocket.error("Error pausing capture", error);
      throw error;
    }
  }
  async resumeCapture() {
    if (!this.recording || !this.paused) return;
    try {
      this.paused = false;
      logger.websocket.info("Audio capture resumed");
    } catch (error) {
      logger.websocket.error("Error resuming capture", error);
      throw error;
    }
  }
  isPaused() {
    return this.paused;
  }
  // ============================================
  // HELPER METHODS
  // ============================================
  async getCurrentMeetingId() {
    try {
      const meetingId = this.meetingService.getCurrentMeetingId();
      if (meetingId) {
        return meetingId;
      }
      const collectedMeetingId = await this.meetingService.collectMeetingId();
      if (collectedMeetingId) {
        return collectedMeetingId;
      }
      return null;
    } catch (error) {
      return null;
    }
  }
  emitTranscriptionEvent(speaker, text, transcriptId, speakerName) {
    const timestamp = (/* @__PURE__ */ new Date()).toISOString();
    const event = new CustomEvent("transcription", {
      detail: {
        speaker,
        speakerName,
        text: text.trim(),
        transcriptId,
        timestamp,
        audioTimestamp: timestamp
      }
    });
    window.dispatchEvent(event);
  }
  emitPartialTranscriptEvent(speaker, text, speakerName) {
    const timestamp = (/* @__PURE__ */ new Date()).toISOString();
    const event = new CustomEvent("partialTranscription", {
      detail: {
        speaker,
        speakerName,
        text: text.trim(),
        timestamp
      }
    });
    window.dispatchEvent(event);
  }
  // ============================================
  // PUBLIC API
  // ============================================
  isRecording() {
    return this.recording;
  }
  getRecordingTime() {
    if (!this.recordingStartTime) return 0;
    return Date.now() - this.recordingStartTime;
  }
  isCaptureEnded() {
    return !this.recording;
  }
  getProcessingStatus() {
    return {
      isProcessing: false,
      isTranscribing: this.recording,
      isAnalyzingAudio: false,
      isProcessingMuteChange: false,
      queueLength: { mic: 0, system: 0 }
    };
  }
}
const websocketAudioService = WebSocketAudioService.getInstance();
window.websocketAudioService = websocketAudioService;
const useAudioRecording = () => {
  const [isRecording, setIsRecording] = reactExports.useState(false);
  const [isPaused, setIsPaused] = reactExports.useState(false);
  const [recordingTime, setRecordingTime] = reactExports.useState(0);
  const [status, setStatus] = reactExports.useState("Ready to start");
  const [error, setError] = reactExports.useState(null);
  const [permissionDenied, setPermissionDenied] = reactExports.useState(false);
  const [isRequestingPermission, setIsRequestingPermission] = reactExports.useState(false);
  const audioService = WebSocketAudioService.getInstance();
  reactExports.useEffect(() => {
    let timer = null;
    if (isRecording && !isPaused) {
      timer = window.setInterval(() => {
        setRecordingTime((prev) => prev + 1);
      }, 1e3);
    } else if (!isRecording) {
      setRecordingTime(0);
    }
    return () => {
      if (timer) {
        clearInterval(timer);
      }
    };
  }, [isRecording, isPaused]);
  const startRecording = reactExports.useCallback(async (userId) => {
    if (isRequestingPermission) {
      return;
    }
    try {
      setError(null);
      setPermissionDenied(false);
      setIsRequestingPermission(true);
      setStatus("Requesting microphone permission...");
      await audioService.startCapture(userId);
      setIsRecording(true);
      setIsPaused(false);
      setRecordingTime(0);
      setStatus("Live transcription active - streaming audio to backend via WebSocket");
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : "Unknown error";
      setError(errorMessage);
      if (errorMessage.includes("cancelled") || errorMessage.includes("denied") || errorMessage.includes("NotAllowedError")) {
        setPermissionDenied(true);
        setStatus("Permission denied - Click Start again and allow microphone access");
      } else if (errorMessage.includes("audio tracks")) {
        setStatus('No audio detected - Please enable "Share tab audio" when selecting the tab');
      } else if (errorMessage.includes("User ID is required")) {
        setStatus("Error: User ID is required to start transcription");
      } else {
        setStatus(`Setup needed - ${errorMessage}`);
      }
    } finally {
      setIsRequestingPermission(false);
    }
  }, [audioService, isRequestingPermission]);
  const stopRecording = reactExports.useCallback(async () => {
    try {
      setStatus("Stopping recording...");
      await audioService.stopCapture();
      setIsRecording(false);
      setIsPaused(false);
      setStatus("Recording stopped");
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : "Unknown error";
      setError(errorMessage);
      setStatus(`Error stopping capture: ${errorMessage}`);
    }
  }, [audioService]);
  const pauseRecording = reactExports.useCallback(async () => {
    try {
      setStatus("Pausing recording...");
      await audioService.pauseCapture();
      setIsPaused(true);
      setStatus("Recording paused - Click resume to continue");
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : "Unknown error";
      setError(errorMessage);
      setStatus(`Error pausing capture: ${errorMessage}`);
    }
  }, [audioService]);
  const resumeRecording = reactExports.useCallback(async () => {
    try {
      setStatus("Resuming recording...");
      await audioService.resumeCapture();
      setIsPaused(false);
      setStatus("Live transcription resumed - streaming audio to backend via WebSocket");
    } catch (err) {
      const errorMessage = err instanceof Error ? err.message : "Unknown error";
      setError(errorMessage);
      setStatus(`Error resuming capture: ${errorMessage}`);
    }
  }, [audioService]);
  const formatTime = reactExports.useCallback((seconds) => {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor(seconds % 3600 / 60);
    const remainingSeconds = seconds % 60;
    if (seconds < 60) {
      return `00:${remainingSeconds.toString().padStart(2, "0")}`;
    }
    if (seconds < 3600) {
      return `${minutes.toString().padStart(2, "0")}:${remainingSeconds.toString().padStart(2, "0")}`;
    }
    return `${hours.toString().padStart(2, "0")}:${minutes.toString().padStart(2, "0")}:${remainingSeconds.toString().padStart(2, "0")}`;
  }, []);
  reactExports.useEffect(() => {
    const handleRecordingStopped = (event) => {
      const { reason } = event.detail;
      setIsRecording(false);
      setIsPaused(false);
      setStatus(`Recording stopped - ${reason === "system_audio_ended" ? "User stopped sharing" : reason === "user_left_call" ? "User left call" : "Stream ended"}`);
      setError(null);
    };
    const handleChromeMessage = (request) => {
      if (request.type === "recordingStoppedByUser") {
        const { reason } = request;
        setIsRecording(false);
        setIsPaused(false);
        setStatus(`Recording stopped - ${reason === "system_audio_ended" ? "User stopped sharing" : reason === "user_left_call" ? "User left call" : "Stream ended"}`);
        setError(null);
      }
    };
    window.addEventListener("recordingStoppedByUser", handleRecordingStopped);
    chrome.runtime.onMessage.addListener(handleChromeMessage);
    return () => {
      window.removeEventListener("recordingStoppedByUser", handleRecordingStopped);
      chrome.runtime.onMessage.removeListener(handleChromeMessage);
    };
  }, []);
  const resetPermissionState = reactExports.useCallback(() => {
    setPermissionDenied(false);
    setError(null);
    setStatus("Ready to start");
  }, []);
  return {
    isRecording,
    isPaused,
    recordingTime,
    status,
    error,
    permissionDenied,
    isRequestingPermission,
    startRecording,
    stopRecording,
    pauseRecording,
    resumeRecording,
    formatTime,
    resetPermissionState
  };
};
class MeetingSaveService {
  static instance;
  static getInstance() {
    if (!MeetingSaveService.instance) {
      MeetingSaveService.instance = new MeetingSaveService();
    }
    return MeetingSaveService.instance;
  }
  /**
   * Save meeting data when transcription starts
   * Returns the API response containing the user ID
   */
  async saveMeeting(payload) {
    try {
      const response = await fetch(ApiEndpoints.meetingSaveEndpoint, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Accept": "application/json"
        },
        body: JSON.stringify(payload),
        mode: "cors",
        cache: "no-cache"
      });
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`Failed to save meeting: ${response.status} - ${errorText}`);
      }
      const data = await response.json();
      if (!data.success) {
        throw new Error(`Save meeting failed: ${data.message}`);
      }
      return data;
    } catch (error) {
      return null;
    }
  }
  /**
   * Update meeting status when transcription stops
   */
  async updateMeetingStatus(payload) {
    try {
      const response = await fetch(ApiEndpoints.meetingUpdateEndpoint, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Accept": "application/json"
        },
        body: JSON.stringify(payload),
        mode: "cors",
        cache: "no-cache"
      });
      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`Failed to update meeting: ${response.status} - ${errorText}`);
      }
      await response.json();
    } catch (error) {
    }
  }
  /**
   * Format date string to "YYYY-MM-DD HH:MM:SS" format expected by the API
   */
  formatDateForAPI(dateString) {
    try {
      if (/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/.test(dateString)) {
        return dateString;
      }
      const date = new Date(dateString);
      if (isNaN(date.getTime())) {
        return this.getCurrentDateTimeForAPI();
      }
      const year = date.getFullYear();
      const month = String(date.getMonth() + 1).padStart(2, "0");
      const day = String(date.getDate()).padStart(2, "0");
      const hours = String(date.getHours()).padStart(2, "0");
      const minutes = String(date.getMinutes()).padStart(2, "0");
      const seconds = String(date.getSeconds()).padStart(2, "0");
      return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    } catch (error) {
      return this.getCurrentDateTimeForAPI();
    }
  }
  /**
   * Get current date/time in API format
   */
  getCurrentDateTimeForAPI() {
    const now = /* @__PURE__ */ new Date();
    const year = now.getFullYear();
    const month = String(now.getMonth() + 1).padStart(2, "0");
    const day = String(now.getDate()).padStart(2, "0");
    const hours = String(now.getHours()).padStart(2, "0");
    const minutes = String(now.getMinutes()).padStart(2, "0");
    const seconds = String(now.getSeconds()).padStart(2, "0");
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
  }
  /**
   * Format attendees list for API
   * Expected format: "email1@example.com (status), email2@example.com (status)"
   */
  formatAttendeesForAPI(attendees) {
    try {
      if (!attendees || !Array.isArray(attendees) || attendees.length === 0) {
        return "None";
      }
      if (typeof attendees === "string") {
        return attendees;
      }
      if (typeof attendees[0] === "object" && attendees[0].email) {
        return attendees.map((attendee) => `${attendee.email} (${attendee.responseStatus || "needsAction"})`).join(", ");
      }
      if (typeof attendees[0] === "string") {
        return attendees.map((email) => `${email} (needsAction)`).join(", ");
      }
      return "None";
    } catch (error) {
      return "None";
    }
  }
  /**
   * Format attachments for API
   */
  formatAttachmentsForAPI(attachments) {
    try {
      if (!attachments) {
        return "None";
      }
      if (typeof attachments === "string") {
        return attachments || "None";
      }
      if (Array.isArray(attachments) && attachments.length > 0) {
        return attachments.map((att) => att.title || att.fileUrl || JSON.stringify(att)).join(", ");
      }
      return "None";
    } catch (error) {
      return "None";
    }
  }
}
const Navigation = React.memo(({
  selectedTab,
  onTabChange,
  disabled = false,
  externalLinkUrl = UrlConfig.webAppUrl,
  meetingId
}) => {
  const { isGlassTransparent } = useGlassTransparent();
  const tabs = reactExports.useMemo(() => [
    { id: "transcripts", label: "Transcripts" },
    { id: "assistant", label: "Pro-active" },
    { id: "coach", label: "Co-pilot" }
  ], []);
  const handleOpenNewTab = reactExports.useCallback(() => {
    let url = externalLinkUrl;
    if (meetingId && meetingId.trim()) {
      url = externalLinkUrl.endsWith("/") ? `${externalLinkUrl}meeting/${meetingId}` : `${externalLinkUrl}/meeting/${meetingId}`;
    }
    if (chrome && chrome.tabs) {
      chrome.tabs.create({ url });
    } else {
      window.open(url, "_blank");
    }
  }, [externalLinkUrl, meetingId]);
  return /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "w-full flex items-center justify-center gap-1 relative", children: [
    tabs.map((tab) => /* @__PURE__ */ jsxRuntimeExports.jsx(
      "button",
      {
        className: `h-8 px-3 text-sm rounded-lg transition-all duration-200 focus:ring-0 focus:ring-offset-0 ${selectedTab === tab.id ? isGlassTransparent ? "bg-transparent text-white font-normal border border-white border-opacity-30 shadow-sm backdrop-blur-sm" : "bg-blue-100 text-blue-700 font-medium" : isGlassTransparent ? "text-white text-opacity-70 hover:bg-white hover:bg-opacity-5 hover:text-white hover:text-opacity-90" : "text-gray-600 hover:bg-gray-100 hover:text-gray-800"} ${disabled ? "opacity-50 cursor-not-allowed" : ""}`,
        onClick: () => !disabled && onTabChange(tab.id),
        disabled,
        children: tab.label
      },
      tab.id
    )),
    /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "absolute right-0 top-0", children: /* @__PURE__ */ jsxRuntimeExports.jsx(
      "button",
      {
        className: `h-8 w-8 flex items-center justify-center rounded-lg transition-colors ${isGlassTransparent ? "hover:bg-white hover:bg-opacity-5" : "hover:bg-gray-100"}`,
        disabled,
        onClick: handleOpenNewTab,
        title: "Open in new tab",
        children: /* @__PURE__ */ jsxRuntimeExports.jsxs(
          "svg",
          {
            width: "16",
            height: "16",
            viewBox: "0 0 24 24",
            fill: "none",
            stroke: "currentColor",
            strokeWidth: "2",
            strokeLinecap: "round",
            strokeLinejoin: "round",
            className: isGlassTransparent ? "text-white text-opacity-70 hover:text-white hover:text-opacity-90" : "text-gray-600 hover:text-gray-800",
            children: [
              /* @__PURE__ */ jsxRuntimeExports.jsx("path", { d: "M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6" }),
              /* @__PURE__ */ jsxRuntimeExports.jsx("polyline", { points: "15,3 21,3 21,9" }),
              /* @__PURE__ */ jsxRuntimeExports.jsx("line", { x1: "10", y1: "14", x2: "21", y2: "3" })
            ]
          }
        )
      }
    ) })
  ] });
});
const MeetingDetails = React.memo(({
  meetingData
}) => {
  const [title, setTitle] = reactExports.useState(meetingData?.title || "");
  const [description, setDescription] = reactExports.useState(meetingData?.description || "");
  const [attachments, setAttachments] = reactExports.useState(meetingData?.attachments || []);
  const { isGlassTransparent } = useGlassTransparent();
  reactExports.useEffect(() => {
    if (meetingData) {
      setTitle(meetingData.title || "");
      setDescription(meetingData.description || "");
      setAttachments(meetingData.attachments || []);
    }
  }, [meetingData]);
  const handleAddAttachment = reactExports.useCallback(() => {
    const newAttachment = prompt("Enter attachment name:");
    if (newAttachment && newAttachment.trim()) {
      setAttachments((prev) => [...prev, newAttachment.trim()]);
    }
  }, []);
  const handleRemoveAttachment = reactExports.useCallback((index) => {
    setAttachments((prev) => prev.filter((_, i) => i !== index));
  }, []);
  return /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "w-full h-full flex flex-col", children: [
    /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: `border-b border-gray-200 px-4 py-3 flex-shrink-0 ${isGlassTransparent ? "glass-transparent-header" : "bg-gray-50"}`, children: /* @__PURE__ */ jsxRuntimeExports.jsx("h1", { className: `text-sm font-medium ${isGlassTransparent ? "glass-transparent-text-bold" : "text-gray-700"}`, children: "Meeting Details" }) }),
    /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "scroll-container p-3 flex-1 overflow-y-scroll", children: [
      /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "mb-3", children: [
        /* @__PURE__ */ jsxRuntimeExports.jsx("label", { className: "block text-xs font-medium text-gray-900 mb-1 text-left", children: "Meeting title" }),
        /* @__PURE__ */ jsxRuntimeExports.jsx(
          "input",
          {
            type: "text",
            value: title,
            onChange: (e) => setTitle(e.target.value),
            className: "w-full px-2 py-1 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-xs",
            placeholder: "Enter meeting title"
          }
        )
      ] }),
      /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "mb-3", children: [
        /* @__PURE__ */ jsxRuntimeExports.jsx("label", { className: "block text-xs font-medium text-gray-900 mb-1 text-left", children: "Description of the meeting" }),
        /* @__PURE__ */ jsxRuntimeExports.jsx(
          "textarea",
          {
            value: description,
            onChange: (e) => setDescription(e.target.value),
            rows: 4,
            className: "w-full px-2 py-1 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none text-xs",
            placeholder: "Enter meeting description"
          }
        )
      ] }),
      /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "mb-3", children: [
        /* @__PURE__ */ jsxRuntimeExports.jsx("label", { className: "block text-xs font-medium text-gray-900 mb-1 text-left", children: "Relevant Attachments" }),
        /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "flex flex-wrap gap-2 mb-2", children: attachments.map((attachment, index) => /* @__PURE__ */ jsxRuntimeExports.jsxs(
          "div",
          {
            className: "flex items-center gap-1 px-2 py-1 bg-gray-200 text-gray-700 rounded-lg text-xs",
            children: [
              /* @__PURE__ */ jsxRuntimeExports.jsx("span", { children: attachment }),
              /* @__PURE__ */ jsxRuntimeExports.jsx(
                "button",
                {
                  onClick: () => handleRemoveAttachment(index),
                  className: "text-gray-500 hover:text-red-500 transition-colors",
                  children: /* @__PURE__ */ jsxRuntimeExports.jsx(Trash2, { size: 14 })
                }
              )
            ]
          },
          index
        )) }),
        /* @__PURE__ */ jsxRuntimeExports.jsxs(
          "button",
          {
            onClick: handleAddAttachment,
            className: "flex items-center gap-1 px-2 py-1 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition-colors text-xs",
            children: [
              /* @__PURE__ */ jsxRuntimeExports.jsx(Upload, { size: 16 }),
              "Upload"
            ]
          }
        )
      ] })
    ] })
  ] });
});
const TranscriptView = React.memo(({
  isVisible,
  transcripts,
  partialTranscript = null,
  isRecording,
  isPaused = false,
  onPause,
  onResume
}) => {
  const [copyButtonText, setCopyButtonText] = reactExports.useState("Copy");
  const [isCopying, setIsCopying] = reactExports.useState(false);
  const { isGlassTransparent } = useGlassTransparent();
  const scrollContainerRef = reactExports.useRef(null);
  const [lastTranscriptCount, setLastTranscriptCount] = reactExports.useState(0);
  reactExports.useEffect(() => {
    if (scrollContainerRef.current) {
      const hasNewTranscript = transcripts.length > lastTranscriptCount;
      const hasPartialTranscript = partialTranscript !== null;
      if (hasNewTranscript || hasPartialTranscript) {
        const container2 = scrollContainerRef.current;
        container2.scrollTop = container2.scrollHeight;
        if (hasNewTranscript) {
          setLastTranscriptCount(transcripts.length);
        }
      }
    }
  }, [transcripts.length, lastTranscriptCount, partialTranscript]);
  reactExports.useEffect(() => {
    if (isVisible && scrollContainerRef.current && transcripts.length > 0) {
      setTimeout(() => {
        if (scrollContainerRef.current) {
          scrollContainerRef.current.scrollTop = scrollContainerRef.current.scrollHeight;
        }
      }, 100);
    }
  }, [isVisible, transcripts.length]);
  const formatTranscriptText = reactExports.useCallback((text) => {
    if (!text) return "";
    let cleanedText = text.trim().replace(/\s+/g, " ").replace(/\n+/g, " ").replace(/\r+/g, " ").replace(/\t+/g, " ").replace(/\s+/g, " ").replace(/\.{3,}/g, "...").replace(/\?{2,}/g, "?").replace(/!{2,}/g, "!").trim();
    return cleanedText;
  }, []);
  const formatTimestamp = reactExports.useCallback((timestamp) => {
    try {
      const date = new Date(timestamp);
      return date.toLocaleTimeString("en-US", {
        hour12: true,
        hour: "2-digit",
        minute: "2-digit",
        second: "2-digit"
      });
    } catch (error) {
      return "--:--:--";
    }
  }, []);
  const getSpeakerInfo = reactExports.useCallback((speakerName) => {
    const hash = speakerName.split("").reduce((acc, char) => acc + char.charCodeAt(0), 0);
    const colorIndex = hash % 8;
    const initials = speakerName.split(" ").map((word) => word.charAt(0).toUpperCase()).join("").substring(0, 2);
    const lightModeColors = [
      { bg: "bg-blue-200", text: "text-blue-800", border: "border-blue-300" },
      { bg: "bg-green-200", text: "text-green-800", border: "border-green-300" },
      { bg: "bg-purple-200", text: "text-purple-800", border: "border-purple-300" },
      { bg: "bg-orange-200", text: "text-orange-800", border: "border-orange-300" },
      { bg: "bg-teal-200", text: "text-teal-800", border: "border-teal-300" },
      { bg: "bg-indigo-200", text: "text-indigo-800", border: "border-indigo-300" },
      { bg: "bg-red-200", text: "text-red-800", border: "border-red-300" },
      { bg: "bg-gray-200", text: "text-gray-800", border: "border-gray-300" }
    ];
    const darkModeColors = [
      { bg: "bg-blue-400", text: "text-blue-900", border: "border-blue-300" },
      { bg: "bg-green-400", text: "text-green-900", border: "border-green-300" },
      { bg: "bg-purple-400", text: "text-purple-900", border: "border-purple-300" },
      { bg: "bg-orange-400", text: "text-orange-900", border: "border-orange-300" },
      { bg: "bg-teal-400", text: "text-teal-900", border: "border-teal-300" },
      { bg: "bg-indigo-400", text: "text-indigo-900", border: "border-indigo-300" },
      { bg: "bg-red-400", text: "text-red-900", border: "border-red-300" },
      { bg: "bg-gray-400", text: "text-gray-900", border: "border-gray-300" }
    ];
    return {
      initials,
      lightModeColor: lightModeColors[colorIndex],
      darkModeColor: darkModeColors[colorIndex]
    };
  }, []);
  const handleCopyTranscript = reactExports.useCallback(async () => {
    if (isCopying) return;
    try {
      setIsCopying(true);
      setCopyButtonText("Copied!");
      const transcriptText = transcripts.map((t) => {
        const speakerName = t.speakerName || (t.speaker === "me" ? "You" : "Speaker");
        const formattedText = formatTranscriptText(t.text);
        const timestamp = formatTimestamp(t.timestamp);
        return `${speakerName} [${timestamp}]: ${formattedText}`;
      }).join("\n\n");
      await navigator.clipboard.writeText(transcriptText);
      setTimeout(() => {
        setCopyButtonText("Copy");
        setIsCopying(false);
      }, 3e3);
    } catch (error) {
      setCopyButtonText("Error");
      setTimeout(() => {
        setCopyButtonText("Copy");
        setIsCopying(false);
      }, 3e3);
    }
  }, [transcripts, formatTranscriptText, formatTimestamp]);
  if (!isVisible) return null;
  return /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "w-full h-full flex flex-col", children: [
    /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: `px-4 py-3 border-b border-gray-200 flex-shrink-0 ${isGlassTransparent ? "glass-transparent-header border-white border-opacity-20" : "bg-gray-50"}`, children: /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex items-center justify-between", children: [
      /* @__PURE__ */ jsxRuntimeExports.jsx("h2", { className: `text-sm font-normal ${isGlassTransparent ? "glass-transparent-text-bold" : "text-gray-800"}`, children: isRecording ? "Live Transcript" : "Transcript" }),
      /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex items-center gap-2", children: [
        isRecording && (onPause || onResume) && /* @__PURE__ */ jsxRuntimeExports.jsxs(
          "button",
          {
            onClick: isPaused ? onResume : onPause,
            className: "solid-button relative group bg-yellow-500 hover:bg-yellow-600 text-white w-7 h-7 rounded-full flex items-center justify-center transition-all duration-300 shadow-md hover:shadow-lg transform hover:scale-105 active:scale-95",
            title: isPaused ? "Resume Recording" : "Pause Recording",
            children: [
              isPaused ? /* @__PURE__ */ jsxRuntimeExports.jsx(Play, { className: "w-3 h-3 ml-0.5 transition-transform duration-200 group-hover:scale-110" }) : /* @__PURE__ */ jsxRuntimeExports.jsx(Pause, { className: "w-3 h-3 transition-transform duration-200 group-hover:scale-110" }),
              /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "absolute inset-0 rounded-full bg-yellow-400 opacity-0 group-hover:opacity-20 transition-opacity duration-300" })
            ]
          }
        ),
        transcripts.length > 0 && /* @__PURE__ */ jsxRuntimeExports.jsx(
          "button",
          {
            onClick: handleCopyTranscript,
            disabled: isCopying,
            className: `px-3 py-1 text-xs rounded-md transition-all duration-200 ${isCopying ? "bg-green-500 text-white cursor-not-allowed" : "bg-blue-500 text-white hover:bg-blue-600"}`,
            title: "Copy transcript to clipboard",
            children: copyButtonText
          }
        ),
        isRecording && /* @__PURE__ */ jsxRuntimeExports.jsxs(jsxRuntimeExports.Fragment, { children: [
          /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "w-2 h-2 bg-red-500 rounded-full animate-pulse" }),
          /* @__PURE__ */ jsxRuntimeExports.jsx("span", { className: `text-xs font-normal ${isGlassTransparent ? "text-red-300" : "text-red-600"} ${isPaused ? "opacity-50" : ""}`, children: isPaused ? "Paused" : "Recording" })
        ] })
      ] })
    ] }) }),
    /* @__PURE__ */ jsxRuntimeExports.jsx(
      "div",
      {
        ref: scrollContainerRef,
        className: "flex-1 overflow-y-auto",
        children: transcripts.length === 0 ? /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "flex flex-col items-center justify-center text-center py-12 px-4", children: isRecording ? /* @__PURE__ */ jsxRuntimeExports.jsxs(jsxRuntimeExports.Fragment, { children: [
          /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "w-8 h-8 mb-4", children: /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "w-full h-full border-2 border-gray-300 border-t-blue-500 rounded-full animate-spin" }) }),
          /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-sm mb-2 font-normal ${isGlassTransparent ? "glass-transparent-text" : "text-gray-600"}`, children: "Listening for speech..." }),
          /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-xs ${isGlassTransparent ? "glass-transparent-text-muted" : "text-gray-500"}`, children: "Start speaking to see transcriptions" })
        ] }) : /* @__PURE__ */ jsxRuntimeExports.jsxs(jsxRuntimeExports.Fragment, { children: [
          /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "w-12 h-12 mb-4 bg-gray-100 rounded-full flex items-center justify-center", children: /* @__PURE__ */ jsxRuntimeExports.jsx("svg", { className: "w-6 h-6 text-gray-400", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsxRuntimeExports.jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 1.5, d: "M19 11a7 7 0 01-7 7m0 0a7 7 0 01-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z" }) }) }),
          /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-sm font-normal mb-1 ${isGlassTransparent ? "glass-transparent-text" : "text-gray-500"}`, children: "No transcripts available" }),
          /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-xs ${isGlassTransparent ? "glass-transparent-text-muted" : "text-gray-400"}`, children: "Start recording to capture meeting audio" })
        ] }) }) : /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "space-y-4 p-4", children: [
          transcripts.map((transcript, index) => {
            const isYou = transcript.speaker === "me";
            const speakerName = transcript.speakerName || (isYou ? "You" : "Speaker");
            const speakerInfo = getSpeakerInfo(speakerName);
            const colorScheme = isGlassTransparent ? speakerInfo.darkModeColor : speakerInfo.lightModeColor;
            return /* @__PURE__ */ jsxRuntimeExports.jsx(
              "div",
              {
                className: `p-4 rounded-xl shadow-sm border ${isGlassTransparent ? "glass-transparent-message" : "bg-white border-gray-200"}`,
                children: /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: `text-sm leading-relaxed ${isGlassTransparent ? "glass-transparent-text" : "text-gray-800"}`, children: [
                  /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex items-center gap-3 mb-3", children: [
                    /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: `flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center font-normal text-sm border-2 ${colorScheme.bg} ${colorScheme.text} ${colorScheme.border} shadow-sm`, children: speakerInfo.initials }),
                    /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex-1 flex items-center justify-between", children: [
                      /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "flex items-center gap-2", children: /* @__PURE__ */ jsxRuntimeExports.jsx("span", { className: `px-2 py-0.5 rounded-full text-xs font-normal ${isGlassTransparent ? "bg-transparent text-white border border-white border-opacity-30" : "bg-blue-50 text-blue-600 border border-blue-200"}`, children: speakerName }) }),
                      /* @__PURE__ */ jsxRuntimeExports.jsx("span", { className: `text-xs font-mono ${isGlassTransparent ? "glass-transparent-text-muted" : "text-gray-500"}`, children: formatTimestamp(transcript.timestamp) })
                    ] })
                  ] }),
                  /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: `ml-13 ${isGlassTransparent ? "glass-transparent-text" : "text-gray-700"}`, children: /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: "leading-relaxed", children: formatTranscriptText(transcript.text) }) })
                ] })
              },
              index
            );
          }),
          partialTranscript && /* @__PURE__ */ jsxRuntimeExports.jsx(
            "div",
            {
              className: `p-4 rounded-xl shadow-sm border animate-pulse ${isGlassTransparent ? "bg-gray-800 bg-opacity-50 border-gray-700 border-opacity-50" : "bg-blue-50 border-blue-200"}`,
              children: /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: `text-sm leading-relaxed ${isGlassTransparent ? "glass-transparent-text" : "text-gray-800"}`, children: [
                /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "flex items-center gap-3 mb-3", children: (() => {
                  const speakerInfo = getSpeakerInfo(partialTranscript.speakerName);
                  const colorScheme = isGlassTransparent ? speakerInfo.darkModeColor : speakerInfo.lightModeColor;
                  return /* @__PURE__ */ jsxRuntimeExports.jsxs(jsxRuntimeExports.Fragment, { children: [
                    /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: `flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center font-normal text-sm border-2 ${colorScheme.bg} ${colorScheme.text} ${colorScheme.border} shadow-sm`, children: speakerInfo.initials }),
                    /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "flex-1 flex items-center justify-between", children: /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex items-center gap-2", children: [
                      /* @__PURE__ */ jsxRuntimeExports.jsx("span", { className: `px-2 py-0.5 rounded-full text-xs font-normal ${isGlassTransparent ? "bg-transparent text-white border border-white border-opacity-30" : "bg-blue-50 text-blue-600 border border-blue-200"}`, children: partialTranscript.speakerName }),
                      /* @__PURE__ */ jsxRuntimeExports.jsx("span", { className: `text-xs italic ${isGlassTransparent ? "text-blue-300" : "text-blue-500"}`, children: "✨ Speaking..." })
                    ] }) })
                  ] });
                })() }),
                /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: `ml-13 ${isGlassTransparent ? "glass-transparent-text" : "text-gray-700"}`, children: /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: "leading-relaxed", children: formatTranscriptText(partialTranscript.text) }) })
              ] })
            }
          )
        ] })
      }
    )
  ] });
});
const GlassTransparentToggle = React.memo(({
  className = "",
  size = "md"
}) => {
  const { isGlassTransparent, toggleGlassTransparent } = useGlassTransparent();
  const iconSizes = {
    sm: "w-4 h-4",
    md: "w-5 h-5",
    lg: "w-6 h-6"
  };
  return /* @__PURE__ */ jsxRuntimeExports.jsx(
    "button",
    {
      onClick: toggleGlassTransparent,
      className: `
        glass-transparent-toggle
        inline-flex items-center justify-center
        focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
        disabled:opacity-50 disabled:pointer-events-none
        ${isGlassTransparent ? "active" : ""}
        ${className}
      `,
      title: isGlassTransparent ? "Disable Glass Transparent" : "Enable Glass Transparent",
      children: /* @__PURE__ */ jsxRuntimeExports.jsxs(
        "svg",
        {
          className: `${iconSizes[size]} transition-transform duration-300 ${isGlassTransparent ? "rotate-180" : ""}`,
          fill: "none",
          stroke: "currentColor",
          viewBox: "0 0 24 24",
          children: [
            /* @__PURE__ */ jsxRuntimeExports.jsx(
              "path",
              {
                strokeLinecap: "round",
                strokeLinejoin: "round",
                strokeWidth: 2,
                d: "M15 12a3 3 0 11-6 0 3 3 0 016 0z"
              }
            ),
            /* @__PURE__ */ jsxRuntimeExports.jsx(
              "path",
              {
                strokeLinecap: "round",
                strokeLinejoin: "round",
                strokeWidth: 2,
                d: "M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
              }
            )
          ]
        }
      )
    }
  );
});
function AppContent() {
  const [show, setShow] = reactExports.useState(false);
  const [selectedTab, setSelectedTab] = reactExports.useState("transcripts");
  const [isInMeeting, setIsInMeeting] = reactExports.useState(false);
  const [isStarting, setIsStarting] = reactExports.useState(false);
  const { isGlassTransparent } = useGlassTransparent();
  const [iconPosition, setIconPosition] = reactExports.useState({ x: 0, y: 0 });
  const [isDragging, setIsDragging] = reactExports.useState(false);
  const [dragOffset, setDragOffset] = reactExports.useState({ x: 0, y: 0 });
  const [hasDragged, setHasDragged] = reactExports.useState(false);
  const [popupPosition, setPopupPosition] = reactExports.useState({ x: 0, y: 0 });
  const [popupDimensions, setPopupDimensions] = reactExports.useState({ width: 400, height: 600, contentHeight: 400 });
  const iconRef = reactExports.useRef(null);
  const [transcripts, setTranscripts] = reactExports.useState([]);
  const [partialTranscript, setPartialTranscript] = reactExports.useState(null);
  const [meetingFormData, setMeetingFormData] = reactExports.useState({
    title: "Product demo call",
    members: ["Jaxon", "Sophie", "Grayson"],
    description: "This session aims to give you a thorough overview of how our solution can optimize your workflow and boost productivity. People will have the chance to ask questions and see the product in action, ensuring you walk away with a solid grasp of its advantages.",
    attachments: [],
    startTime: "",
    endTime: "",
    meetLink: "",
    organizerEmail: "",
    status: "",
    created: "",
    updated: "",
    google_event_id: `temp_${Date.now()}`,
    // Temporary Google Calendar event ID (will be replaced when meeting is loaded)
    organizer: "",
    dialIns: []
  });
  const [currentMeetingId, setCurrentMeetingId] = reactExports.useState("");
  const { isUserAuthenticated, user, isSignedIn } = useAuth();
  const {
    isRecording,
    isPaused,
    recordingTime,
    startRecording,
    stopRecording,
    pauseRecording,
    resumeRecording,
    formatTime
  } = useAudioRecording();
  const meetingService = MeetingService.getInstance();
  const meetingSaveService = MeetingSaveService.getInstance();
  const checkIfInMeeting = reactExports.useCallback(() => {
    const pathname = window.location.pathname;
    const meetingIdPattern = /\/[a-z0-9]{3,4}-[a-z0-9]{4}-[a-z0-9]{3,4}(?:\?|$)/;
    const lookupPattern = /\/lookup\/[a-z0-9-]+/;
    const isInActiveMeeting = meetingIdPattern.test(pathname) || lookupPattern.test(pathname);
    setIsInMeeting(isInActiveMeeting);
    return isInActiveMeeting;
  }, []);
  const detectChromeUIElements = reactExports.useCallback(() => {
    const chromeUI = {
      topNotificationBar: 0,
      bottomNotificationBar: 0,
      leftSidebar: 0,
      rightSidebar: 0
    };
    try {
      const topElements = document.querySelectorAll('[role="banner"], .chrome-notification, [data-chrome-notification]');
      let maxTopHeight = 0;
      topElements.forEach((element) => {
        const rect = element.getBoundingClientRect();
        if (rect.top >= 0 && rect.top < 100 && rect.height > 0) {
          maxTopHeight = Math.max(maxTopHeight, rect.height);
        }
      });
      const allElements = document.querySelectorAll("*");
      allElements.forEach((element) => {
        const rect = element.getBoundingClientRect();
        const computedStyle = window.getComputedStyle(element);
        if (rect.top >= 0 && rect.top < 60 && rect.height >= 30 && rect.height <= 80) {
          const hasNotificationText = element.textContent?.includes("sharing") || element.textContent?.includes("Stop sharing") || element.textContent?.includes("Screen sharing") || element.textContent?.includes("Stop") || element.textContent?.includes("Share") || element.textContent?.includes("meet.google.com") || element.textContent?.includes("tab") || element.textContent?.includes("window");
          const hasNotificationStyle = computedStyle.backgroundColor !== "rgba(0, 0, 0, 0)" || computedStyle.background !== "none" || computedStyle.border !== "none" || computedStyle.position === "fixed" || computedStyle.position === "absolute";
          const isChromeNotification = element.tagName === "DIV" && (computedStyle.zIndex === "auto" || parseInt(computedStyle.zIndex) > 1e3) && rect.width >= window.innerWidth * 0.8;
          if (hasNotificationText || hasNotificationStyle || isChromeNotification) {
            maxTopHeight = Math.max(maxTopHeight, rect.height);
          }
        }
      });
      if (maxTopHeight > 0) {
        chromeUI.topNotificationBar = Math.max(maxTopHeight, 40);
      } else {
        const viewportHeight = window.innerHeight;
        const screenHeight = window.screen.height;
        if (screenHeight - viewportHeight > 50) {
          chromeUI.topNotificationBar = 50;
        }
      }
      const bottomElements = document.querySelectorAll('[role="status"], .chrome-status, [data-chrome-status]');
      let maxBottomHeight = 0;
      bottomElements.forEach((element) => {
        const rect = element.getBoundingClientRect();
        if (rect.bottom >= window.innerHeight - 50 && rect.height > 0) {
          maxBottomHeight = Math.max(maxBottomHeight, rect.height);
        }
      });
      if (maxBottomHeight > 0) {
        chromeUI.bottomNotificationBar = maxBottomHeight;
      }
      return chromeUI;
    } catch (error) {
      return chromeUI;
    }
  }, []);
  const calculateResponsiveDimensions = reactExports.useCallback(() => {
    const viewportWidth = window.innerWidth;
    const viewportHeight = window.innerHeight;
    const chromeUI = detectChromeUIElements();
    const topChromeUIHeight = chromeUI.topNotificationBar;
    const bottomChromeUIHeight = chromeUI.bottomNotificationBar;
    const availableHeight = viewportHeight - topChromeUIHeight - bottomChromeUIHeight;
    let popupWidth;
    if (viewportWidth <= 800) {
      popupWidth = Math.min(350, viewportWidth - 24);
    } else if (viewportWidth <= 1200) {
      popupWidth = Math.min(380, viewportWidth - 40);
    } else {
      popupWidth = Math.min(400, viewportWidth - 60);
    }
    let popupHeight;
    const minHeight = 300;
    const maxHeight = 700;
    const preferredHeight = Math.min(600, availableHeight - 40);
    if (availableHeight < 400) {
      popupHeight = Math.max(minHeight, availableHeight - 20);
    } else if (availableHeight < 600) {
      popupHeight = Math.max(minHeight, availableHeight - 30);
    } else {
      popupHeight = Math.min(maxHeight, preferredHeight);
    }
    const headerHeight = 120;
    const contentHeight = Math.max(200, popupHeight - headerHeight);
    return {
      popupWidth,
      popupHeight,
      contentHeight,
      availableHeight,
      chromeUI
    };
  }, [detectChromeUIElements]);
  const calculatePopupPosition = reactExports.useCallback((iconX, iconY) => {
    const iconSize = 48;
    const margin = 12;
    const dimensions = calculateResponsiveDimensions();
    const { popupWidth, popupHeight, availableHeight, chromeUI } = dimensions;
    const topChromeUIHeight = chromeUI.topNotificationBar;
    const bottomChromeUIHeight = chromeUI.bottomNotificationBar;
    const availableTop = topChromeUIHeight + margin;
    const availableBottom = window.innerHeight - bottomChromeUIHeight - margin;
    let popupX = iconX;
    let popupY = iconY - popupHeight - margin;
    if (iconX + popupWidth > window.innerWidth) {
      popupX = window.innerWidth - popupWidth - margin;
    }
    if (popupX < margin) {
      popupX = margin;
    }
    if (popupY < availableTop) {
      popupY = iconY + iconSize + margin;
      if (popupY + popupHeight > availableBottom) {
        popupX = iconX - popupWidth - margin;
        popupY = iconY;
        if (popupX < margin) {
          popupX = iconX + iconSize + margin;
        }
        if (popupX + popupWidth > window.innerWidth - margin) {
          popupX = window.innerWidth - popupWidth - margin;
        }
      }
    }
    if (popupY + popupHeight > availableBottom) {
      popupY = availableBottom - popupHeight;
    }
    if (popupY < availableTop) {
      popupY = availableTop;
    }
    if (popupHeight > availableHeight) {
      popupY = availableTop;
    }
    return {
      x: popupX,
      y: popupY,
      width: popupWidth,
      height: popupHeight,
      contentHeight: dimensions.contentHeight
    };
  }, [calculateResponsiveDimensions]);
  const toggle = (e) => {
    if (isDragging || hasDragged) {
      e?.preventDefault();
      return;
    }
    if (!show) {
      const newPopupData = calculatePopupPosition(iconPosition.x, iconPosition.y);
      setPopupPosition({ x: newPopupData.x, y: newPopupData.y });
      setPopupDimensions({
        width: newPopupData.width,
        height: newPopupData.height,
        contentHeight: newPopupData.contentHeight
      });
    }
    setShow(!show);
  };
  reactExports.useEffect(() => {
    setIconPosition({ x: window.innerWidth - 68, y: window.innerHeight - 68 });
  }, []);
  const saveIconPosition = reactExports.useCallback((_position) => {
  }, []);
  const handleMouseDown = reactExports.useCallback((e) => {
    if (e.button !== 0) return;
    e.preventDefault();
    setIsDragging(true);
    setHasDragged(false);
    const rect = iconRef.current?.getBoundingClientRect();
    if (rect) {
      setDragOffset({
        x: e.clientX - rect.left,
        y: e.clientY - rect.top
      });
    }
  }, []);
  const handleMouseMove = reactExports.useCallback((e) => {
    if (!isDragging) return;
    e.preventDefault();
    const newX = e.clientX - dragOffset.x;
    const newY = e.clientY - dragOffset.y;
    setHasDragged(true);
    const iconSize = 48;
    const constrainedX = Math.max(0, Math.min(newX, window.innerWidth - iconSize));
    const constrainedY = Math.max(0, Math.min(newY, window.innerHeight - iconSize));
    setIconPosition({ x: constrainedX, y: constrainedY });
    if (show) {
      const newPopupData = calculatePopupPosition(constrainedX, constrainedY);
      setPopupPosition({ x: newPopupData.x, y: newPopupData.y });
      setPopupDimensions({
        width: newPopupData.width,
        height: newPopupData.height,
        contentHeight: newPopupData.contentHeight
      });
    }
  }, [isDragging, dragOffset, show, calculatePopupPosition]);
  const handleMouseUp = reactExports.useCallback(() => {
    if (isDragging) {
      setIsDragging(false);
      saveIconPosition(iconPosition);
      setTimeout(() => {
        setHasDragged(false);
      }, 100);
    }
  }, [isDragging, iconPosition, saveIconPosition]);
  reactExports.useEffect(() => {
    if (isDragging) {
      document.addEventListener("mousemove", handleMouseMove);
      document.addEventListener("mouseup", handleMouseUp);
      return () => {
        document.removeEventListener("mousemove", handleMouseMove);
        document.removeEventListener("mouseup", handleMouseUp);
      };
    }
  }, [isDragging, handleMouseMove, handleMouseUp]);
  reactExports.useEffect(() => {
    const handleResize = () => {
      const iconSize = 48;
      const constrainedX = Math.max(0, Math.min(iconPosition.x, window.innerWidth - iconSize));
      const constrainedY = Math.max(0, Math.min(iconPosition.y, window.innerHeight - iconSize));
      if (constrainedX !== iconPosition.x || constrainedY !== iconPosition.y) {
        const newPosition = { x: constrainedX, y: constrainedY };
        setIconPosition(newPosition);
        saveIconPosition(newPosition);
      }
      if (show) {
        const newPopupData = calculatePopupPosition(iconPosition.x, iconPosition.y);
        setPopupPosition({ x: newPopupData.x, y: newPopupData.y });
        setPopupDimensions({
          width: newPopupData.width,
          height: newPopupData.height,
          contentHeight: newPopupData.contentHeight
        });
      }
    };
    window.addEventListener("resize", handleResize);
    return () => window.removeEventListener("resize", handleResize);
  }, [iconPosition, saveIconPosition, show, calculatePopupPosition]);
  reactExports.useEffect(() => {
    checkIfInMeeting();
    const handleUrlChange = () => {
      setTimeout(() => {
        const wasInMeeting = isInMeeting;
        const nowInMeeting = checkIfInMeeting();
        if (!wasInMeeting && nowInMeeting) {
          setIconPosition({ x: window.innerWidth - 68, y: window.innerHeight - 68 });
        }
      }, 100);
    };
    window.addEventListener("popstate", handleUrlChange);
    const originalPushState = history.pushState;
    const originalReplaceState = history.replaceState;
    history.pushState = function(...args) {
      originalPushState.apply(history, args);
      handleUrlChange();
    };
    history.replaceState = function(...args) {
      originalReplaceState.apply(history, args);
      handleUrlChange();
    };
    return () => {
      window.removeEventListener("popstate", handleUrlChange);
      history.pushState = originalPushState;
      history.replaceState = originalReplaceState;
    };
  }, [checkIfInMeeting, isInMeeting]);
  reactExports.useEffect(() => {
    if (!show) return;
    const checkChromeUIChanges = () => {
      const newPopupData = calculatePopupPosition(iconPosition.x, iconPosition.y);
      setPopupPosition({ x: newPopupData.x, y: newPopupData.y });
      setPopupDimensions({
        width: newPopupData.width,
        height: newPopupData.height,
        contentHeight: newPopupData.contentHeight
      });
    };
    const interval = setInterval(checkChromeUIChanges, 2e3);
    const handleVisibilityChange = () => {
      if (!document.hidden) {
        setTimeout(checkChromeUIChanges, 100);
      }
    };
    document.addEventListener("visibilitychange", handleVisibilityChange);
    return () => {
      clearInterval(interval);
      document.removeEventListener("visibilitychange", handleVisibilityChange);
    };
  }, [show, iconPosition, calculatePopupPosition]);
  reactExports.useEffect(() => {
    const handleTranscription = (event) => {
      const { speaker, speakerName, text, transcriptId, timestamp, audioTimestamp } = event.detail;
      const actualTimestamp = audioTimestamp || timestamp;
      setTranscripts((prev) => {
        const newTranscripts = [...prev, {
          speaker,
          speakerName: speakerName || (speaker === "me" ? "You" : "Speaker"),
          // Fallback for backward compatibility
          text,
          transcriptId,
          timestamp: actualTimestamp
        }];
        return newTranscripts.sort(
          (a, b) => new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime()
        );
      });
      setPartialTranscript(null);
    };
    const handlePartialTranscription = (event) => {
      const { speaker, speakerName, text } = event.detail;
      logger.app.debug("Received partial transcription event", { speaker, speakerName, text });
      setPartialTranscript({
        speaker,
        speakerName: speakerName || (speaker === "me" ? "You" : "Speaker"),
        text
      });
    };
    window.addEventListener("transcription", handleTranscription);
    window.addEventListener("partialTranscription", handlePartialTranscription);
    return () => {
      window.removeEventListener("transcription", handleTranscription);
      window.removeEventListener("partialTranscription", handlePartialTranscription);
    };
  }, [user?.email]);
  reactExports.useEffect(() => {
    const handleRecordingStopped = async (event) => {
      const { reason } = event.detail;
      if (isRecording) {
        await stopRecording();
        const meetingId = meetingService.getCurrentMeetingId();
        if (meetingId && user?.email) {
          try {
            const googleEventId = meetingFormData.google_event_id && meetingFormData.google_event_id.trim() !== "" ? meetingFormData.google_event_id : `manual_${meetingId}_${Date.now()}`;
            const updateMeetingPayload = {
              email: user.email,
              meeting_id: meetingId,
              meeting_status: "completed",
              google_event_id: googleEventId
            };
            await meetingSaveService.updateMeetingStatus(updateMeetingPayload);
          } catch (error) {
          }
        }
        setCurrentMeetingId("");
        setMeetingFormData((prev) => ({
          ...prev,
          status: reason === "system_audio_ended" ? "User stopped sharing" : reason === "user_left_call" ? "User left call" : "Recording stopped"
        }));
        setSelectedTab("transcripts");
      }
    };
    const handleChromeMessage = async (request) => {
      if (request.type === "recordingStoppedByUser") {
        const { reason } = request;
        if (isRecording) {
          await stopRecording();
          const meetingId = meetingService.getCurrentMeetingId();
          if (meetingId && user?.email) {
            try {
              const googleEventId = meetingFormData.google_event_id && meetingFormData.google_event_id.trim() !== "" ? meetingFormData.google_event_id : `manual_${meetingId}_${Date.now()}`;
              const updateMeetingPayload = {
                email: user.email,
                meeting_id: meetingId,
                meeting_status: "completed",
                google_event_id: googleEventId
              };
              await meetingSaveService.updateMeetingStatus(updateMeetingPayload);
            } catch (error) {
            }
          }
          setCurrentMeetingId("");
          setMeetingFormData((prev) => ({
            ...prev,
            status: reason === "system_audio_ended" ? "User stopped sharing" : reason === "user_left_call" ? "User left call" : "Recording stopped"
          }));
          setSelectedTab("transcripts");
        }
      }
    };
    window.addEventListener("recordingStoppedByUser", handleRecordingStopped);
    const chromeMessageHandler = (request, _sender, _sendResponse) => {
      handleChromeMessage(request);
      return true;
    };
    chrome.runtime.onMessage.addListener(chromeMessageHandler);
    return () => {
      window.removeEventListener("recordingStoppedByUser", handleRecordingStopped);
      chrome.runtime.onMessage.removeListener(chromeMessageHandler);
    };
  }, [isRecording, stopRecording, meetingService, user?.email, meetingFormData.google_event_id, meetingSaveService]);
  const loadMeetingInfo = reactExports.useCallback(async () => {
    try {
      if (!user?.email) {
        return;
      }
      const currentMeetingId2 = await collectMeetingId();
      if (currentMeetingId2) {
        chrome.storage.local.set({ currentMeetingId: currentMeetingId2 });
        await loadMeetingDetailsFromCalendar(currentMeetingId2);
      }
    } catch (error) {
    }
  }, [user?.email]);
  const collectMeetingId = async () => {
    try {
      if (window.location && window.location.href) {
        const currentUrl = window.location.href;
        const meetUrlPatterns = [
          /https:\/\/meet\.google\.com\/([a-z\-]+)/i,
          /https:\/\/meet\.google\.com\/([a-z\-]{3}-[a-z\-]{4}-[a-z\-]{3})/i,
          /meet\.google\.com\/([a-zA-Z\-]{10,})/i
        ];
        for (const pattern of meetUrlPatterns) {
          const match = currentUrl.match(pattern);
          if (match && match[1]) {
            return match[1];
          }
        }
      }
      try {
        const response = await new Promise((resolve, reject) => {
          const timeout = setTimeout(() => {
            reject(new Error("Background script timeout"));
          }, 3e3);
          chrome.runtime.sendMessage({ type: "MEETING_ID_IN_BG" }, (response2) => {
            clearTimeout(timeout);
            if (chrome.runtime.lastError) {
              reject(new Error(chrome.runtime.lastError.message));
            } else {
              resolve(response2 || {});
            }
          });
        });
        if (response?.meetingId) {
          return response.meetingId;
        }
      } catch (bgError) {
      }
      try {
        const { tabUrl } = await new Promise((resolve, reject) => {
          const timeout = setTimeout(() => {
            reject(new Error("Tab info timeout"));
          }, 2e3);
          chrome.runtime.sendMessage({ action: "checkTab" }, (response) => {
            clearTimeout(timeout);
            if (chrome.runtime.lastError) {
              reject(new Error(chrome.runtime.lastError.message));
            } else {
              resolve(response || {});
            }
          });
        });
        if (tabUrl) {
          const meetUrlMatch = tabUrl.match(/https:\/\/meet\.google\.com\/([a-z\-]+)/i);
          if (meetUrlMatch && meetUrlMatch[1]) {
            return meetUrlMatch[1];
          }
        }
      } catch (tabError) {
      }
      return null;
    } catch (error) {
      return null;
    }
  };
  reactExports.useEffect(() => {
    if (isSignedIn && user?.email) {
      loadMeetingInfo();
      collectMeetingId().then((id) => {
        if (id) {
          setCurrentMeetingId(id);
        }
      });
    }
  }, [isSignedIn, user?.email, loadMeetingInfo]);
  const loadMeetingDetailsFromCalendar = async (meetingId) => {
    try {
      if (!user?.email) return;
      const { token, error } = await new Promise((resolve) => {
        chrome.runtime.sendMessage({ action: "getToken" }, (response) => {
          if (chrome.runtime.lastError) {
            resolve({ error: chrome.runtime.lastError.message });
          } else {
            resolve(response || {});
          }
        });
      });
      if (error || !token) {
        setMeetingFormData((prev) => ({
          ...prev,
          title: meetingId,
          description: "Calendar access not available - please sign in again",
          startTime: "Unknown",
          endTime: "Unknown",
          meetLink: `https://meet.google.com/${meetingId}`,
          organizerEmail: "Unknown",
          status: "Active",
          created: (/* @__PURE__ */ new Date()).toLocaleString(),
          updated: (/* @__PURE__ */ new Date()).toLocaleString(),
          members: []
        }));
        return;
      }
      const calendarEvents = await fetchTodaysCalendarEvents(token, meetingId);
      const meetEvents = calendarEvents.filter(
        (e) => e.hangoutLink || e.conferenceData?.conferenceSolution?.name === "Google Meet"
      );
      if (meetEvents.length > 0) {
        let meetingDetail = null;
        meetingDetail = meetEvents.find(
          (m) => m?.conferenceData?.conferenceId?.includes(meetingId)
        );
        if (!meetingDetail) {
          meetingDetail = meetEvents.find(
            (m) => m.hangoutLink && m.hangoutLink.includes(meetingId)
          );
        }
        if (!meetingDetail) {
          meetingDetail = meetEvents.find((m) => {
            if (m.hangoutLink) {
              const urlMatch = m.hangoutLink.match(/meet\.google\.com\/([a-z\-]+)/i);
              return urlMatch && urlMatch[1] === meetingId;
            }
            return false;
          });
        }
        if (meetingDetail) {
          const extractedData = extractMeetingDetails(meetingDetail);
          setMeetingFormData((prev) => ({
            ...prev,
            ...extractedData
          }));
        } else {
          const fallbackGoogleEventId = `manual_${meetingId}_${Date.now()}`;
          setMeetingFormData((prev) => ({
            ...prev,
            title: meetingId,
            description: "No calendar details found for this meeting",
            startTime: "Unknown",
            endTime: "Unknown",
            meetLink: `https://meet.google.com/${meetingId}`,
            organizerEmail: "Unknown",
            status: "Active",
            created: (/* @__PURE__ */ new Date()).toLocaleString(),
            updated: (/* @__PURE__ */ new Date()).toLocaleString(),
            members: [],
            google_event_id: fallbackGoogleEventId
          }));
        }
      } else {
        const fallbackGoogleEventId = `manual_${meetingId}_${Date.now()}`;
        setMeetingFormData((prev) => ({
          ...prev,
          title: meetingId,
          description: "No calendar details found for this meeting",
          startTime: "Unknown",
          endTime: "Unknown",
          meetLink: `https://meet.google.com/${meetingId}`,
          organizerEmail: "Unknown",
          status: "Active",
          created: (/* @__PURE__ */ new Date()).toLocaleString(),
          updated: (/* @__PURE__ */ new Date()).toLocaleString(),
          members: [],
          google_event_id: fallbackGoogleEventId
        }));
      }
    } catch (error) {
    }
  };
  const extractMeetingDetails = (event) => {
    const start = new Date(event.start?.dateTime || event.start?.date).toLocaleString();
    const end = new Date(event.end?.dateTime || event.end?.date).toLocaleString();
    const meetLink = event.hangoutLink || event.conferenceData?.entryPoints?.find((ep) => ep.entryPointType === "video")?.uri || "";
    const attendees = event.attendees?.map((a) => a.email).join(", ") || "";
    return {
      title: event.summary || "Untitled Meeting",
      description: event.description || "",
      startTime: start,
      endTime: end,
      meetLink,
      organizerEmail: event.organizer?.email || "",
      status: event.status || "",
      created: new Date(event.created).toLocaleString(),
      updated: new Date(event.updated).toLocaleString(),
      members: attendees ? attendees.split(", ") : [],
      google_event_id: event.id
      // Google Calendar event ID (required)
    };
  };
  const fetchTodaysCalendarEvents = async (token, _meetingId) => {
    try {
      const now = /* @__PURE__ */ new Date();
      const startOfDay = new Date(now);
      startOfDay.setHours(0, 0, 0, 0);
      const endOfDay = new Date(now);
      endOfDay.setHours(23, 59, 59, 999);
      const timeMin = startOfDay.toISOString();
      const timeMax = endOfDay.toISOString();
      const res = await fetch(
        `https://www.googleapis.com/calendar/v3/calendars/primary/events?orderBy=startTime&singleEvents=true&timeMin=${encodeURIComponent(timeMin)}&timeMax=${encodeURIComponent(timeMax)}&conferenceDataVersion=1&maxResults=50`,
        {
          headers: {
            Authorization: `Bearer ${token}`
          }
        }
      );
      if (!res.ok) {
        const err = await res.json().catch(() => ({ error: { message: "Unknown error" } }));
        if (res.status === 401) {
          chrome.storage.local.remove("userInfo");
          throw new Error("Authentication expired - please sign in again");
        }
        throw new Error(`Google Calendar API Error: ${res.status} - ${err.error?.message || "Unknown error"}`);
      }
      const data = await res.json();
      return data.items || [];
    } catch (error) {
      return [];
    }
  };
  const handleTabChange = (tab) => {
    setSelectedTab(tab);
  };
  const handleStart = reactExports.useCallback(async () => {
    if (!user?.email) {
      return;
    }
    try {
      setIsStarting(true);
      let meetingId = meetingService.getCurrentMeetingId();
      if (!meetingId) {
        meetingId = await meetingService.collectMeetingId();
      }
      if (!meetingId) {
        throw new Error("Failed to collect meeting ID from Google Meet");
      }
      loadMeetingInfo().catch((_error) => {
      });
      const googleEventId = meetingFormData.google_event_id && meetingFormData.google_event_id.trim() !== "" ? meetingFormData.google_event_id : `manual_${meetingId}_${Date.now()}`;
      const saveMeetingPayload = {
        email: user.email,
        meetingId,
        google_event_id: googleEventId,
        // Required: Google Calendar event ID
        startTime: meetingSaveService.formatDateForAPI(meetingFormData.startTime || (/* @__PURE__ */ new Date()).toLocaleString()),
        endTime: meetingSaveService.formatDateForAPI(meetingFormData.endTime || (/* @__PURE__ */ new Date()).toLocaleString()),
        attendees: meetingSaveService.formatAttendeesForAPI(meetingFormData.members || []),
        attachments: meetingSaveService.formatAttachmentsForAPI(meetingFormData.attachments),
        meetLink: meetingFormData.meetLink || `https://meet.google.com/${meetingId}`,
        description: meetingFormData.description || "",
        title: meetingFormData.title || "Meeting",
        organizerEmail: meetingFormData.organizerEmail || user.email,
        status: meetingFormData.status || "confirmed",
        created: meetingSaveService.formatDateForAPI(meetingFormData.created || (/* @__PURE__ */ new Date()).toLocaleString()),
        updated: meetingSaveService.formatDateForAPI(meetingFormData.updated || (/* @__PURE__ */ new Date()).toLocaleString())
      };
      const saveResponse = await meetingSaveService.saveMeeting(saveMeetingPayload);
      if (!saveResponse) {
        throw new Error("Save meeting API returned null response");
      }
      if (!saveResponse.success) {
        throw new Error(`Save meeting failed: ${saveResponse.message}`);
      }
      if (!saveResponse.data || !saveResponse.data.user) {
        throw new Error(`Save meeting API response missing user ID. Response: ${JSON.stringify(saveResponse)}`);
      }
      const userId = saveResponse.data.user;
      await startRecording(userId);
    } catch (error) {
    } finally {
      setIsStarting(false);
    }
  }, [user?.email, startRecording, loadMeetingInfo, meetingFormData, meetingService, meetingSaveService]);
  const handleStop = async () => {
    const meetingId = meetingService.getCurrentMeetingId();
    logger.app.info("handleStop called", {
      meetingId,
      userEmail: user?.email,
      googleEventId: meetingFormData.google_event_id
    });
    try {
      meetingService.resetMeetingState();
      await stopRecording();
      logger.app.debug("Checking conditions for API call", {
        hasMeetingId: !!meetingId,
        hasUserEmail: !!user?.email
      });
      if (meetingId && user?.email) {
        try {
          const googleEventId = meetingFormData.google_event_id && meetingFormData.google_event_id.trim() !== "" ? meetingFormData.google_event_id : `manual_${meetingId}_${Date.now()}`;
          const updateMeetingPayload = {
            email: user.email,
            meeting_id: meetingId,
            meeting_status: "completed",
            google_event_id: googleEventId
          };
          logger.app.info("Calling updateMeetingStatus API", updateMeetingPayload);
          await meetingSaveService.updateMeetingStatus(updateMeetingPayload);
          logger.app.info("API call successful");
        } catch (error) {
          logger.app.error("API call failed", error);
        }
      } else {
        logger.app.warn("Skipping API call - missing meetingId or user email");
      }
      setCurrentMeetingId("");
      setMeetingFormData((prev) => ({
        ...prev,
        // Keep all existing meeting details, just update status to indicate session ended
        status: "Recording Stopped"
      }));
      setSelectedTab("transcripts");
    } catch (error) {
      logger.app.error("Error in handleStop", error);
    }
  };
  const handleGearClick = () => {
    setSelectedTab("details");
  };
  reactExports.useEffect(() => {
    chrome.storage.local.get(["meetingFormData"], (result) => {
      if (result.meetingFormData) {
        setMeetingFormData((prev) => ({
          ...prev,
          ...result.meetingFormData
        }));
      }
    });
  }, []);
  const isAuthenticated = isUserAuthenticated() || isSignedIn;
  if (!isInMeeting) {
    return null;
  }
  const logoUrl = typeof chrome !== "undefined" && chrome.runtime && typeof chrome.runtime.getURL === "function" ? chrome.runtime.getURL("src/assets/images/techjays_logo.jpg") : "/src/assets/images/techjays_logo.jpg";
  return /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: `popup-container ${isGlassTransparent ? "glass-mode-active" : ""}`, style: {
    position: "fixed",
    left: `${iconPosition.x}px`,
    top: `${iconPosition.y}px`,
    right: "auto",
    bottom: "auto",
    zIndex: 9999
  }, children: [
    show && /* @__PURE__ */ jsxRuntimeExports.jsx(
      "div",
      {
        className: `popup-content ${show ? "opacity-100" : "opacity-0"} ${isGlassTransparent ? "glass-transparent-popup" : ""}`,
        style: {
          width: `${popupDimensions.width}px`,
          height: `${popupDimensions.height}px`,
          maxHeight: `${popupDimensions.height}px`,
          overflowY: "auto",
          position: "fixed",
          left: `${popupPosition.x}px`,
          top: `${popupPosition.y}px`,
          right: "auto",
          bottom: "auto"
        },
        children: !isAuthenticated ? (
          // Authentication Notice
          /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "w-full flex items-center justify-center py-8", children: /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "bg-white rounded-lg p-6 shadow-lg border border-gray-200 max-w-md text-center", children: /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "mb-4", children: [
            /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4", children: /* @__PURE__ */ jsxRuntimeExports.jsx("svg", { className: "w-8 h-8 text-blue-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsxRuntimeExports.jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" }) }) }),
            /* @__PURE__ */ jsxRuntimeExports.jsx("h3", { className: "text-lg font-semibold text-gray-900 mb-2", children: "Meeting Coach" }),
            /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: "text-gray-600 text-sm mb-4", children: "Please sign in with your Techjays account to use Meeting Coach features" }),
            /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "mb-4 flex justify-center", children: /* @__PURE__ */ jsxRuntimeExports.jsx(
              GlassTransparentToggle,
              {
                size: "sm",
                className: "p-2 hover:bg-gray-100 rounded-md transition-colors"
              }
            ) }),
            /* @__PURE__ */ jsxRuntimeExports.jsx(
              "button",
              {
                className: "bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-md text-sm font-medium transition-colors",
                onClick: () => {
                },
                children: "🔑 Sign In via Extension Popup"
              }
            ),
            /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: "text-xs text-gray-500 mt-3", children: "Click the Meeting Coach extension icon in your browser toolbar to sign in" })
          ] }) }) })
        ) : (
          // Main Interface (when authenticated)
          /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "w-full h-full flex flex-col space-y-4", style: { height: `${popupDimensions.height - 40}px` }, children: [
            /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: `${isGlassTransparent ? "glass-transparent-card" : "bg-gray-50"} rounded-xl p-4 shadow-lg`, children: [
              /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex items-center mb-4 gap-3", children: [
                /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex items-center gap-2 flex-1 min-w-0", children: [
                  /* @__PURE__ */ jsxRuntimeExports.jsx(
                    "h1",
                    {
                      className: `text-lg font-semibold text-gray-800 truncate ${isGlassTransparent ? "glass-transparent-text" : ""}`,
                      title: meetingFormData.title || "Meeting Title",
                      children: meetingFormData.title || "Meeting Title"
                    }
                  ),
                  /* @__PURE__ */ jsxRuntimeExports.jsx(
                    "button",
                    {
                      className: "p-1 hover:bg-gray-100 rounded-md transition-colors flex-shrink-0",
                      title: "Meeting Details & Settings",
                      onClick: handleGearClick,
                      children: /* @__PURE__ */ jsxRuntimeExports.jsx(
                        Settings,
                        {
                          className: `h-4 w-4 transition-colors ${selectedTab === "details" ? "text-blue-600" : "text-gray-600 hover:text-gray-800"}`
                        }
                      )
                    }
                  )
                ] }),
                /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: "flex items-center gap-2 flex-shrink-0", children: [
                  /* @__PURE__ */ jsxRuntimeExports.jsx(
                    GlassTransparentToggle,
                    {
                      size: "sm",
                      className: "p-2 hover:bg-gray-100 rounded-md transition-colors"
                    }
                  ),
                  isRecording && /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: `flex items-center gap-1 text-gray-700 bg-gray-100 px-2 py-1 rounded-md ${isGlassTransparent ? "glass-transparent-status" : ""}`, children: [
                    /* @__PURE__ */ jsxRuntimeExports.jsx(Clock, { className: "w-3 h-3" }),
                    /* @__PURE__ */ jsxRuntimeExports.jsx("span", { className: `text-xs font-medium ${isGlassTransparent ? "glass-transparent-text" : ""}`, children: formatTime(recordingTime) })
                  ] }),
                  isRecording ? /* @__PURE__ */ jsxRuntimeExports.jsxs(
                    "button",
                    {
                      className: "solid-button relative group bg-red-500 hover:bg-red-600 text-white w-12 h-12 rounded-full flex items-center justify-center transition-all duration-300 disabled:opacity-50 disabled:cursor-not-allowed shadow-lg hover:shadow-xl transform hover:scale-105 active:scale-95",
                      onClick: handleStop,
                      title: "Stop Recording",
                      disabled: !isAuthenticated,
                      children: [
                        /* @__PURE__ */ jsxRuntimeExports.jsx(Square, { className: "w-5 h-5 transition-transform duration-200 group-hover:scale-110" }),
                        /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "absolute inset-0 rounded-full bg-red-400 opacity-0 group-hover:opacity-20 transition-opacity duration-300" })
                      ]
                    }
                  ) : /* @__PURE__ */ jsxRuntimeExports.jsx(
                    "button",
                    {
                      className: "solid-button relative group bg-blue-500 hover:bg-blue-600 text-white w-12 h-12 rounded-full flex items-center justify-center transition-all duration-300 disabled:opacity-50 disabled:cursor-not-allowed shadow-lg hover:shadow-xl transform hover:scale-105 active:scale-95",
                      onClick: handleStart,
                      title: !isAuthenticated ? "Please sign in to start recording" : "Start Recording",
                      disabled: !isAuthenticated || isStarting,
                      children: isStarting ? /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" }) : /* @__PURE__ */ jsxRuntimeExports.jsxs(jsxRuntimeExports.Fragment, { children: [
                        /* @__PURE__ */ jsxRuntimeExports.jsx(Play, { className: "w-5 h-5 ml-0.5 transition-transform duration-200 group-hover:scale-110" }),
                        /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "absolute inset-0 rounded-full bg-blue-400 opacity-0 group-hover:opacity-20 transition-opacity duration-300" })
                      ] })
                    }
                  )
                ] })
              ] }),
              /* @__PURE__ */ jsxRuntimeExports.jsx(
                Navigation,
                {
                  selectedTab,
                  onTabChange: handleTabChange,
                  disabled: !isAuthenticated,
                  externalLinkUrl: UrlConfig.webAppUrl,
                  meetingId: currentMeetingId
                }
              )
            ] }),
            /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { className: `rounded-xl shadow-lg flex-1 min-h-0 overflow-hidden ${isGlassTransparent ? "glass-transparent-card" : "bg-white"}`, style: { height: `${popupDimensions.contentHeight}px` }, children: [
              selectedTab === "details" && /* @__PURE__ */ jsxRuntimeExports.jsx(
                MeetingDetails,
                {
                  meetingData: {
                    title: meetingFormData.title,
                    attendees: meetingFormData.members.join(", "),
                    description: meetingFormData.description,
                    attachments: meetingFormData.attachments,
                    meetLink: meetingFormData.meetLink
                  }
                }
              ),
              selectedTab === "transcripts" && /* @__PURE__ */ jsxRuntimeExports.jsx(
                TranscriptView,
                {
                  isVisible: selectedTab === "transcripts",
                  transcripts,
                  partialTranscript,
                  isRecording,
                  isPaused,
                  onPause: pauseRecording,
                  onResume: resumeRecording
                }
              ),
              selectedTab === "assistant" && /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "p-8 text-center h-full flex items-center justify-center", children: /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { children: [
                /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "mb-4", children: /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "inline-flex items-center justify-center w-16 h-16 rounded-full bg-blue-100 mb-4", children: /* @__PURE__ */ jsxRuntimeExports.jsx("svg", { className: "w-8 h-8 text-blue-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsxRuntimeExports.jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M13 10V3L4 14h7v7l9-11h-7z" }) }) }) }),
                /* @__PURE__ */ jsxRuntimeExports.jsx("h3", { className: `text-lg font-semibold text-gray-800 mb-2 ${isGlassTransparent ? "glass-transparent-text" : ""}`, children: "Proactive Assistant" }),
                /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-sm text-gray-600 ${isGlassTransparent ? "glass-transparent-text-muted" : ""}`, children: "Coming Soon" }),
                /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-xs text-gray-500 mt-2 ${isGlassTransparent ? "glass-transparent-text-muted" : ""}`, children: "AI-powered meeting assistance will be available soon" })
              ] }) }),
              selectedTab === "coach" && /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "p-8 text-center h-full flex items-center justify-center", children: /* @__PURE__ */ jsxRuntimeExports.jsxs("div", { children: [
                /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "mb-4", children: /* @__PURE__ */ jsxRuntimeExports.jsx("div", { className: "inline-flex items-center justify-center w-16 h-16 rounded-full bg-purple-100 mb-4", children: /* @__PURE__ */ jsxRuntimeExports.jsx("svg", { className: "w-8 h-8 text-purple-600", fill: "none", stroke: "currentColor", viewBox: "0 0 24 24", children: /* @__PURE__ */ jsxRuntimeExports.jsx("path", { strokeLinecap: "round", strokeLinejoin: "round", strokeWidth: 2, d: "M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z" }) }) }) }),
                /* @__PURE__ */ jsxRuntimeExports.jsx("h3", { className: `text-lg font-semibold text-gray-800 mb-2 ${isGlassTransparent ? "glass-transparent-text" : ""}`, children: "Co-Pilot Coach" }),
                /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-sm text-gray-600 ${isGlassTransparent ? "glass-transparent-text-muted" : ""}`, children: "Coming Soon" }),
                /* @__PURE__ */ jsxRuntimeExports.jsx("p", { className: `text-xs text-gray-500 mt-2 ${isGlassTransparent ? "glass-transparent-text-muted" : ""}`, children: "Real-time meeting coaching will be available soon" })
              ] }) })
            ] })
          ] })
        )
      }
    ),
    /* @__PURE__ */ jsxRuntimeExports.jsx(
      "button",
      {
        ref: iconRef,
        className: `toggle-button ${isDragging ? "dragging" : ""}`,
        onClick: toggle,
        onMouseDown: handleMouseDown,
        style: {
          cursor: isDragging ? "grabbing" : "grab",
          transform: isDragging ? "scale(1.1)" : "scale(1)",
          transition: isDragging ? "none" : "all 0.2s ease"
        },
        children: /* @__PURE__ */ jsxRuntimeExports.jsx(
          "img",
          {
            src: logoUrl,
            alt: "Meeting Coach",
            className: "button-icon"
          }
        )
      }
    )
  ] });
}
function App() {
  return /* @__PURE__ */ jsxRuntimeExports.jsx(GlassTransparentProvider, { children: /* @__PURE__ */ jsxRuntimeExports.jsx(AppContent, {}) });
}
const container = document.createElement("div");
container.id = "crxjs-app";
container.style.position = "fixed";
container.style.top = "0";
container.style.left = "0";
container.style.zIndex = "999999";
container.style.pointerEvents = "none";
document.body.appendChild(container);
clientExports.createRoot(container).render(
  /* @__PURE__ */ jsxRuntimeExports.jsx(reactExports.StrictMode, { children: /* @__PURE__ */ jsxRuntimeExports.jsx(App, {}) })
);
//# sourceMappingURL=index.tsx-C4_Pj31v.js.map
